#pragma once

#include <ymod_pq/call.h>
#include <yplatform/time_traits.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>
#include <yplatform/log/contains_logger.h>
#include <charconv>

namespace ymod_pq {

class replica_monitor
    : public std::enable_shared_from_this<replica_monitor>
    , public yplatform::log::contains_logger
{
    using yield_context = yplatform::yield_context<replica_monitor>;
    using duration = yplatform::time_traits::duration;
    using seconds = yplatform::time_traits::seconds;
    using clock = yplatform::time_traits::clock;
    struct list_handler;

public:
    struct settings
    {
        std::string query;
        duration poll_interval = seconds(5);
        duration replica_list_ttl = seconds(10);
    };

    replica_monitor(
        const yplatform::task_context_ptr& ctx,
        const yplatform::log::source& logger,
        boost::asio::io_service& io,
        const std::string& db,
        const call_ptr& pq,
        const settings& settings)
        : yplatform::log::contains_logger(logger)
        , strand_(io)
        , timer_(io)
        , db_(db)
        , pq_(pq)
        , settings_(settings)
        , ctx_(ctx)
        , running_(false)
    {
    }

    struct replica
    {
        std::string host;
        duration lag = seconds(0);
    };

    std::vector<replica> get() const
    {
        auto now = clock::now();
        std::lock_guard<std::mutex> l(lock_);
        return now < replicas_expiry_ ? replicas_ : std::vector<replica>();
    }

    void start()
    {
        running_ = true;
        yplatform::spawn(strand_, shared_from_this());
    }

    void stop()
    {
        running_ = false;
    }

    void operator()(yield_context yield_ctx, std::vector<replica> replicas = {})
    {
        reenter(yield_ctx)
        {
            while (running_)
            {
                yield list_replicas(yield_ctx);
                if (replicas.size() > 0)
                {
                    set(std::move(replicas));
                }
                yield wait(yield_ctx, settings_.poll_interval);
            }
        }
    }

    // Pq handler.
    void operator()(
        yield_context yield_ctx,
        const future_result& future,
        const boost::shared_ptr<list_handler>& list_handler)
    {
        try
        {
            future.get();
            operator()(yield_ctx, std::move(list_handler->replicas));
        }
        catch (const std::exception& e)
        {
            YLOG_CTX_LOCAL(ctx_, error) << "replica monitor query fail: " << e.what();
            operator()(yield_ctx, {});
        }
    }

private:
    struct list_handler : public response_handler
    {
        enum column : unsigned
        {
            host = 0,
            lag,

            cnt
        };

        virtual unsigned column_count() const override
        {
            return (unsigned)column::cnt;
        }

        virtual void handle_row_begin(unsigned /*row*/) override
        {
            skip_ = false;
            current_ = {};
        }

        virtual void handle_cell(
            unsigned /*row*/,
            unsigned col,
            const std::string& value,
            bool is_null) override
        {
            if (skip_)
            {
                return;
            }
            if (is_null || value.empty())
            {
                skip_ = true;
                return;
            }
            switch (col)
            {
            case column::host:
            {
                current_.host = value;
                break;
            }
            case column::lag:
            {
                current_.lag = parse_usec(value);
                break;
            }
            default:
                // Ignore unexpected columns.
                break;
            }
        }

        virtual void handle_row_end(unsigned /*row*/) override
        {
            if (!skip_)
            {
                replicas.push_back(current_);
            }
        }

        std::vector<replica> replicas;

    private:
        duration parse_usec(const std::string& value)
        {
            uint64_t usec = 0;
            auto [p, ec] = std::from_chars(value.begin(), value.end(), usec);
            skip_ = ec != std::errc();
            return yplatform::time_traits::microseconds(usec);
        }

        replica current_;
        bool skip_;
    };

    void list_replicas(yield_context yield_ctx)
    {
        auto pq_handler = boost::make_shared<list_handler>();
        auto future = pq_->request(ctx_, db_, settings_.query, nullptr, pq_handler, false);
        future.add_callback([future, pq_handler, yield_ctx]() { yield_ctx(future, pq_handler); });
    }

    void wait(yield_context yield_ctx, duration d)
    {
        timer_.expires_from_now(d);
        timer_.async_wait([yield_ctx](const boost::system::error_code&) { yield_ctx(); });
    }

    void set(std::vector<replica> replicas)
    {
        auto now = clock::now();
        std::lock_guard<std::mutex> l(lock_);
        replicas_ = std::move(replicas);
        replicas_expiry_ = now + settings_.replica_list_ttl;
    }

    boost::asio::io_service::strand strand_;
    yplatform::time_traits::timer timer_;
    std::string db_;
    call_ptr pq_;
    settings settings_;

    mutable std::mutex lock_;
    std::vector<replica> replicas_;
    yplatform::time_traits::time_point replicas_expiry_;

    yplatform::task_context_ptr ctx_;
    std::atomic<bool> running_;
};

using replica_monitor_ptr = std::shared_ptr<replica_monitor>;

}

#include <yplatform/unyield.h>
