#ifndef _YMOD_PQ_SETTINGS_H_
#define _YMOD_PQ_SETTINGS_H_

#include "query_conf.h"
#include <yplatform/time_traits.h>
#include <yplatform/ptree.h>
#include <apq/connection_pool.hpp>

namespace ymod_pq {

struct settings
{
    settings() = default;
    settings(const yplatform::ptree& xml)
    {
        parse(xml);
    }

    query_conf queries;
    yplatform::time_traits::duration idle_timeout = yplatform::time_traits::duration::max();
    yplatform::time_traits::duration queue_timeout = yplatform::time_traits::duration::max();
    yplatform::time_traits::duration connect_timeout = yplatform::time_traits::duration::max();
    std::size_t max_conn = 500;
    std::string conninfo;
    bool async_resolve = true;
    bool ipv6_only = true;
    bool log_pa = false;
    yplatform::time_traits::duration default_deadline = yplatform::time_traits::duration::max();
    apq::connection_pool::stat_window_size_t stat_window_size = 10000;
    yplatform::net::dns::resolver_service_settings dns = { .cache_ttl = std::chrono::minutes(5) };

    void parse(const yplatform::ptree& xml)
    {
        using time_duration = yplatform::time_traits::duration;
        using seconds = yplatform::time_traits::seconds;

        queries.parse(xml.get("query_conf", ""));
        log_pa = xml.get("log_pa", log_pa ? 1 : 0);
        auto queue_timeout_ = xml.get("queue_timeout", time_duration(seconds(5)));
        if (queue_timeout_.count() >= 0) queue_timeout = queue_timeout_;
        auto connect_timeout_ = xml.get("connect_timeout", time_duration(seconds(5)));
        if (connect_timeout_.count() >= 0) connect_timeout = connect_timeout_;
        auto default_deadline_ = xml.get("default_deadline", time_duration(seconds(-1)));
        if (default_deadline_.count() >= 0) default_deadline = default_deadline_;
        max_conn = xml.get<std::size_t>("max_conn", 10);
        async_resolve = xml.get("async_resolve", async_resolve);
        ipv6_only = xml.get("ipv6_only", ipv6_only);
        stat_window_size =
            xml.get<apq::connection_pool::stat_window_size_t>("stat_window_size", stat_window_size);
        if (stat_window_size <= 0)
            throw std::invalid_argument(
                "ymod_pq invalid stat_window_size " + std::to_string(stat_window_size));
        conninfo = xml.get("conninfo", "");
        if (auto dns_ptree = xml.get_child_optional("dns")) dns.parse(*dns_ptree);
    }
};

typedef boost::shared_ptr<settings> settings_ptr;

}

#endif // _YMOD_PQ_SETTINGS_H_
