#include <yplatform/log.h>
#include <ymod_python/module.h>
#include <ymod_python/object.h>
#include <mail/ymod_python/src/ymod_python/impl.h>

namespace ymod_python {

namespace {

using namespace boost::python;

class current_python {
public:
    current_python() {
        python_impl = std::dynamic_pointer_cast<impl>(get_current_python());
    }
    void set_as_current() {
        python_impl->register_python_in_tls();
    }
private:
    std::shared_ptr<impl> python_impl;
};

void deadline_timer(PyObject* id_raw, long time, PyObject* cb_raw)
{
    if (!PyCallable_Check(cb_raw)) {
        PyErr_SetString(PyExc_TypeError, "ymod_python_sys::deadline_timer expects a callable object");
        throw_error_already_set();
    }

    auto python = get_current_python();
    auto& io = python->get_io_service();
    auto timer = std::make_shared<boost::asio::deadline_timer>(io, boost::posix_time::milliseconds(time));

    PyObjectShared id_shared{id_raw};
    PyObjectShared cb_shared{cb_raw};

    timer->async_wait([id_shared, cb_shared, timer, python](const boost::system::error_code& ec){
        if (!ec) {
            python->async_run([id_shared, cb_shared] {
                try {
                    object cb = cb_shared;
                    object id = id_shared;
                    cb(id);
                } catch (const error_already_set&) {
                    PyErr_Print();
                    L_(error) << "Exception in ymod_python_sys.deadline_timer callback.";
                }
            });
        }
    });
}

void terminate_application()
{
    kill(getpid(), SIGTERM);
}

void log(yplatform::log::severity_level level, str s_)
{
    using namespace yplatform::log;
    extract<std::string> s(s_);
    switch (level) {
        case emerg: L_(emerg) << std::string(s()); break;
        case alert: L_(alert) << std::string(s()); break;
        case fatal: L_(fatal) << std::string(s()); break;
        case error: L_(error) << std::string(s()); break;
        case warning: L_(warning) << std::string(s()); break;
        case notice: L_(notice) << std::string(s()); break;
        case info: L_(info) << std::string(s()); break;
        case debug: L_(debug) << std::string(s()); break;
        default:
            L_(warning) << "wrong log level of next message, default to warning:";
            L_(warning) << std::string(s());
    }
}

BOOST_PYTHON_MODULE(ymod_python_sys)
{
    class_<current_python, boost::shared_ptr<current_python>>("current_python", init<>())
        .def("set_as_current", &current_python::set_as_current);

    def("deadline_timer", deadline_timer, (arg("id"), arg("timeout"), arg("callback")));
    def("terminate_application", terminate_application);

    enum_<yplatform::log::severity_level>("severity_level")
        .value("emerg", yplatform::log::emerg)
        .value("alert", yplatform::log::alert)
        .value("fatal", yplatform::log::fatal)
        .value("error", yplatform::log::error)
        .value("warning", yplatform::log::warning)
        .value("notice", yplatform::log::notice)
        .value("info", yplatform::log::info)
        .value("debug", yplatform::log::debug)
        .export_values();

    def("log", log);
}

}
}
