#include "impl.h"
#include <ymod_python/errors.h>

#include <memory>
#include <future>
#include <boost/thread/tss.hpp>

namespace ymod_python {

namespace {
    // We do not care of TLS objects life-time because:
    // 1. Python PyThreadStates are destroyed in fini() of main_python
    //    when all worker threads are stopped, so nobody
    //    will execute python code.
    // 2. All impl pointers are stored in user code as shared_ptr, so
    //    they are not hang over in TLS, and nobody will use
    //    them after stop() stage
    boost::thread_specific_ptr<impl> tls_python([](impl*){});
    boost::thread_specific_ptr<PyThreadState> tls_PyThreadState([](PyThreadState*){});
    python_ptr master_python;
}

impl::impl(yplatform::reactor& reactor)
    : reactor_(reactor)
    , py_main_thread_(nullptr)
{
}

python_ptr get_current_python() {
    if (tls_python.get() != nullptr) {
        return std::dynamic_pointer_cast<python>(tls_python.get()->shared_from_this());
    }
    return master_python;
}

boost::asio::io_service& impl::get_io_service() {
    return *(reactor_.io());
}

void impl::register_python_in_tls() {
    if (tls_python.get()) {
        tls_python.reset(this);
    }
}

void impl::register_pyThreadState_in_tls() {
    if (tls_PyThreadState.get() == nullptr) {
#ifdef USE_PYTHON3
        tls_PyThreadState.reset(PyThreadState_New(PyInterpreterState_Main()));
#else
        tls_PyThreadState.reset(PyThreadState_New(PyInterpreterState_Head()));
#endif
    }
}

void impl::init(const yplatform::ptree& config) {
    py_source_ = config.get<std::string>("load", "");
    if(!Py_IsInitialized()) {
        master_python = std::dynamic_pointer_cast<python>(shared_from_this());
#ifdef USE_PYTHON3
        program_name_ = Py_DecodeLocale(config.get<std::string>("program_name").c_str(), NULL);
        python_home_ = Py_DecodeLocale(config.get<std::string>("python_home").c_str(), NULL);
        python_path_ = Py_DecodeLocale(config.get<std::string>("python_path").c_str(), NULL);
        Py_SetProgramName(program_name_);
        Py_SetPythonHome(python_home_);
        Py_InitializeEx(0);
        PySys_SetPath(python_path_);
#else
        program_name_ = config.get<std::string>("program_name","ymod_python");
        python_home_ = config.get<std::string>("python_home","/");
        python_path_ = config.get<std::string>("python_path","/");
        Py_SetProgramName(const_cast<char*>(program_name_.c_str()));
        Py_SetPythonHome(const_cast<char*>(python_home_.c_str()));
        Py_InitializeEx(0);
        PyEval_InitThreads();
        PySys_SetPath(const_cast<char*>(python_path_.c_str()));
#endif
        py_main_thread_ = PyEval_SaveThread();
    }
}

void impl::start() {
    if (!py_source_.empty()) {
        std::string file_name = py_source_;
        async_run([file_name] {
            FILE *file = fopen(file_name.c_str(), "r");
            if (file) {
                PyObject* main = PyImport_ImportModule("__main__");
                PyObject* globals = PyObject_GetAttrString(main, "__dict__");
                PyObject* result = PyRun_FileEx(file, file_name.c_str(), Py_file_input, globals, globals, 1);
                if (result == nullptr &&  PyErr_Occurred()) {
                    L_(error) << "Cannot load: " + file_name + ", reason: python exception during load.";
                    PyErr_Print();
                }
                Py_XDECREF(result);
                Py_DECREF(main);
                Py_DECREF(globals);
            } else {
                L_(error) << "Cannot load: " + file_name + ", reason: file not found";
            }
        });
    }
}

void impl::reload(const yplatform::ptree& config) {
    //TODO: check if reload is needed
    py_source_ = config.get<std::string>("load", "");
    //TODO: call reload callback in python
    start();
}

void impl::fini() {
    if(py_main_thread_) {
        PyThreadState* st = PyGILState_GetThisThreadState();
        if (st == nullptr) {
            PyEval_RestoreThread(py_main_thread_);
            Py_Finalize();
        } else {
            L_(warning) << "Incorrect ymod_python termination";
        }
        master_python.reset();
#ifdef USE_PYTHON3
        PyMem_RawFree(program_name_);
        PyMem_RawFree(python_home_);
        PyMem_RawFree(python_path_);
#endif
    }
}

void impl::sync_run(handler fn) {
    sync_run([&fn](auto){
        fn();
    });
}

void impl::async_run(handler fn) {
    async_run([fn](auto){
        fn();
    });
}

void impl::sync_run(handler_with_gil_releaser fn) {
    boost::asio::io_service& io_service_ = get_io_service();
    if (io_service_.get_executor().running_in_this_thread()) {
        register_python_in_tls();
        register_pyThreadState_in_tls();
        PyEval_RestoreThread(tls_PyThreadState.get());
        try {
            fn([](auto f){
                PyEval_SaveThread();
                try {
                    f();
                } catch (const std::exception& e) {
                    PyEval_RestoreThread(tls_PyThreadState.get());
                    L_(error) << "Unhandled exception in ymod_python::sync_run inner function: " + std::string{e.what()};
                    throw;
                } catch (...) {
                    PyEval_RestoreThread(tls_PyThreadState.get());
                    L_(error) << "Unhandled exception in ymod_python::sync_run inner function";
                    throw;
                }
                PyEval_RestoreThread(tls_PyThreadState.get());
            });
        } catch (const std::exception& e) {
            PyEval_SaveThread();
            L_(error) << "Unhandled exception in ymod_python::sync_run: " + std::string{e.what()};
            throw;
        } catch (...) {
            PyEval_SaveThread();
            L_(error) << "Unhandled exception in ymod_python::sync_run";
            throw;
        }
        PyEval_SaveThread();
    } else {
        std::promise<void> p;
        boost::asio::post(io_service_, [this, &fn, &p] {
            register_python_in_tls();
            register_pyThreadState_in_tls();
            PyEval_RestoreThread(tls_PyThreadState.get());
            try {
                fn([](auto f){
                    PyEval_SaveThread();
                    try {
                        f();
                    } catch (const std::exception& e) {
                        PyEval_RestoreThread(tls_PyThreadState.get());
                        L_(error) << "Unhandled exception in ymod_python::sync_run inner function: " + std::string{e.what()};
                        throw;
                    } catch (...) {
                        PyEval_RestoreThread(tls_PyThreadState.get());
                        L_(error) << "Unhandled exception in ymod_python::sync_run inner function";
                        throw;
                    }
                    PyEval_RestoreThread(tls_PyThreadState.get());
                });
            } catch (const std::exception& e) {
                PyEval_SaveThread();
                L_(error) << "Unhandled exception in ymod_python::sync_run: " + std::string{e.what()};
                try {
                    p.set_exception(std::current_exception());
                } catch (...) {
                    L_(error) << "Unhandled exception in ymod_python::sync_run at promise.set_exception";
                }
                return;
            } catch (...) {
                PyEval_SaveThread();
                L_(error) << "Unhandled exception in ymod_python::sync_run";
                try {
                    p.set_exception(std::current_exception());
                } catch (...) {
                    L_(error) << "Unhandled exception in ymod_python::sync_run at promise.set_exception";
                }
                return;
            }
            PyEval_SaveThread();
            p.set_value();
        });
        p.get_future().wait();
    }
}

void impl::async_run(handler_with_gil_releaser fn) {
    boost::asio::io_service& io_service_ = get_io_service();
    boost::asio::post(io_service_, [this, fn] {
        register_python_in_tls();
        register_pyThreadState_in_tls();
        PyEval_RestoreThread(tls_PyThreadState.get());
        try {
            fn([](auto f){
                PyEval_SaveThread();
                try {
                    f();
                } catch (const std::exception& e) {
                    L_(error) << "Unhandled exception in ymod_python::async_run inner function: " + std::string{e.what()};
                } catch (...) {
                    L_(error) << "Unhandled exception in ymod_python::async_run inner function";
                }
                PyEval_RestoreThread(tls_PyThreadState.get());
            });
        } catch (const std::exception& e) {
            L_(error) << "Unhandled exception in ymod_python::async_run: " + std::string{e.what()};
        } catch (...) {
            L_(error) << "Unhandled exception in ymod_python::async_run";
        }
        PyEval_SaveThread();
    });
}

}

#include <yplatform/module_registration.h>
REGISTER_MODULE(ymod_python::impl)
