#pragma once

#include <mail/ymod_queuedb/include/types.h>


namespace ymod_queuedb {

struct Queue {
    virtual ~Queue() { }

    template<class Context = io_result::sync_context>
    auto acquireTasks(const Worker& worker, TasksLimit tasksLimit,
                      const RequestId& reqId, Context ctx = io_result::use_sync) const {
        io_result::detail::init_async_result<Context, OnAcquireTasks> init{ctx};
        acquireTasksAsync(worker, tasksLimit, reqId, init.handler);
        return init.result.get();
    }

    template<class Context = io_result::sync_context>
    auto addTask(Uid uid, const TaskType& task, const TaskArgs& taskArgs, Timeout timeout,
                 const RequestId& reqId, Context ctx = io_result::use_sync) const {
        io_result::detail::init_async_result<Context, OnAddTask> init{ctx};
        addTaskAsync(uid, task, taskArgs, timeout, reqId, init.handler);
        return init.result.get();
    }

    template<class Context = io_result::sync_context>
    auto completeTask(TaskId taskId, const Worker& worker, const RequestId& reqId,
                      Context ctx = io_result::use_sync) const {
        io_result::detail::init_async_result<Context, OnExecute> init{ctx};
        completeTaskAsync(taskId, worker, reqId, init.handler);
        return init.result.get();
    }

    template<class Context = io_result::sync_context>
    auto failTask(TaskId taskId, const Worker& worker, const Reason& reason, MaxRetries maxRetries,
                  Delay delay, const RequestId& reqId, Context ctx = io_result::use_sync) const {
        io_result::detail::init_async_result<Context, OnExecute> init{ctx};
        failTaskAsync(taskId, worker, reason, maxRetries, delay, reqId, init.handler);
        return init.result.get();
    }

    template<class Context = io_result::sync_context>
    auto delayTask(TaskId taskId, const Worker& worker, Delay delay,  const RequestId& reqId,
                   Context ctx = io_result::use_sync) const {
        io_result::detail::init_async_result<Context, OnExecute> init{ctx};
        delayTaskAsync(taskId, worker, delay, reqId, init.handler);
        return init.result.get();
    }

    template<class Context = io_result::sync_context>
    auto refreshTask(TaskId taskId, const Worker& worker, const RequestId& reqId,
                     Context ctx = io_result::use_sync) const {
        io_result::detail::init_async_result<Context, OnExecute> init{ctx};
        refreshTaskAsync(taskId, worker, reqId, init.handler);
        return init.result.get();
    }

protected:
    virtual void acquireTasksAsync(const Worker& worker, TasksLimit tasksLimit,
                                   const RequestId& reqId, OnAcquireTasks cb) const = 0;
    virtual void addTaskAsync(Uid uid, const TaskType& task, const TaskArgs& taskArgs,
                              Timeout timeout, const RequestId& reqId,  OnAddTask cb) const = 0;
    virtual void completeTaskAsync(TaskId taskId, const Worker& worker, const RequestId& reqId, OnExecute cb) const = 0;
    virtual void failTaskAsync(TaskId taskId, const Worker& worker, const Reason& reason, MaxRetries maxRetries,
                               Delay delay, const RequestId& reqId, OnExecute cb) const = 0;
    virtual void delayTaskAsync(TaskId taskId, const Worker& worker, Delay delay,
                                const RequestId& reqId, OnExecute cb) const = 0;
    virtual void refreshTaskAsync(TaskId taskId, const Worker& worker,
                                  const RequestId& reqId, OnExecute cb) const = 0;
};

using QueuePtr = std::shared_ptr<Queue>;

}
