#include <mail/ymod_queuedb/include/queue.h>
#include <mail/ymod_queuedb/include/error.h>
#include <mail/ymod_queuedb/include/internal/query.h>
#include <mail/ymod_queuedb/include/internal/logger.h>
#include <mail/ymod_queuedb/include/internal/state.h>
#include <yamail/data/deserialization/ptree.h>
#include <yamail/data/serialization/json_writer.h>
#include <mail/webmail/commondb/include/with_executers.h>
#include <mail/webmail/commondb/include/settings_reflection.h>
#include <yplatform/module_registration.h>
#include <pgg/database/pool_factory.h>
#include <pgg/factory.h>
#include <fstream>

BOOST_FUSION_DEFINE_STRUCT((ymod_queuedb), Settings,
    (std::string, reactor)
    (std::string, logger)
)

namespace ymod_queuedb {

using namespace commondb;
using namespace std::string_literals;

auto convert(result::Task v) {
    Task result;
    result.taskId = v.task_id;
    result.uid = v.uid;
    result.service = std::move(v.service);
    result.task = std::move(v.task);
    result.state = ymod_queuedb::pg::TaskState::fromString(v.state);
    result.taskArgs = (v.task_args ? std::move(*v.task_args): std::string());
    result.reassignmentCount = v.reassignment_count;
    result.tries = v.tries;
    result.tryNotices = std::move(v.try_notices);
    result.created = v.created;
    result.processingDate = v.processing_date;
    result.timeoutSec = std::chrono::seconds(v.timeout_sec);
    result.worker = std::move(v.worker);
    result.requestId = std::move(v.request_id);

    return result;
}

struct Postgres: public WithExecuters, public Queue, public yplatform::module {

    virtual ~Postgres() { }

    void init(const yplatform::ptree& cfg) {
        common = yamail::data::deserialization::fromPtree<Settings>(cfg);
        pg = readPgSettings(cfg.get_child("pg"));

        initExecuters(common.reactor, pg, query::QueriesRegister(), query::ParametersRegister());

        LOGDOG_(getModuleLogger(common.logger), notice,
            log::message="ymod_queuedb::Postgres module initialized",
            log::settings=yamail::data::serialization::JsonWriter(common).result(),
            log::pg_settings=yamail::data::serialization::JsonWriter(pg).result(),
            log::password_length=passwordSize()
        );
    }

    using WithExecuters::readWriteExecutor;
    auto readWriteExecutor(const std::string& requestId) const {
        auto ctxLogger = getContextLogger(common.logger, RequestId(requestId));
        auto logger = makePggLogger(ctxLogger);
        auto profiler = commondb::pggProfiler("", requestId, "queuedb");
        return readWriteExecutor(requestId, logger, profiler);
    }

    void acquireTasksAsync(const Worker& worker, TasksLimit tasksLimit, const RequestId& requestId, OnAcquireTasks cb) const  override {
        readWriteExecutor(requestId)->request(
            pgg::withQuery<query::AcquireTasks>(
                query::Worker(worker),
                query::TasksLimit(tasksLimit)
            ),
            std::move(cb),
            [] (result::Task v) {
                return convert(std::move(v));
            }
        );
    }

    void addTaskAsync(Uid uid, const TaskType& task, const TaskArgs& taskArgs, Timeout timeout, const RequestId& requestId, OnAddTask cb) const override {
        readWriteExecutor(requestId)->request(
            pgg::withQuery<query::AddTask>(
                query::Uid(uid),
                query::TaskType(task),
                query::TaskArgs(taskArgs),
                query::Timeout(timeout),
                query::RequestId(requestId)
            ),
            std::move(cb),
            [] (result::AddTask v) {
                return TaskId(v.task_id);
            }
        );
    }

    void completeTaskAsync(TaskId taskId, const Worker& worker, const RequestId& requestId, OnExecute cb) const override {
        readWriteExecutor(requestId)->execute(
            pgg::withQuery<query::CompleteTask>(
                query::TaskId(taskId),
                query::Worker(worker)
            ),
            std::move(cb)
        );
    }

    void failTaskAsync(TaskId taskId, const Worker& worker, const Reason& reason, MaxRetries maxRetries, Delay delay, const RequestId& requestId, OnExecute cb) const override {
        readWriteExecutor(requestId)->execute(
            pgg::withQuery<query::FailTask>(
                query::TaskId(taskId),
                query::Worker(worker),
                query::Reason(reason),
                query::MaxRetries(maxRetries),
                query::Delay(delay)
            ),
            std::move(cb)
        );
    }

    void delayTaskAsync(TaskId taskId, const Worker& worker, Delay delay, const RequestId& requestId, OnExecute cb) const override {
        readWriteExecutor(requestId)->execute(
            pgg::withQuery<query::DelayTask>(
                query::TaskId(taskId),
                query::Worker(worker),
                query::Delay(delay)
            ),
            std::move(cb)
        );
    }

    void refreshTaskAsync(TaskId taskId, const Worker& worker, const RequestId& requestId, OnExecute cb) const override {
        readWriteExecutor(requestId)->execute(
            pgg::withQuery<query::RefreshTask>(
                query::TaskId(taskId),
                query::Worker(worker)
            ),
            std::move(cb)
        );
    }

    Settings common;
    PgSettings pg;
};

}

DEFINE_SERVICE_OBJECT(ymod_queuedb::Postgres)
