#include <yplatform/find.h>
#include <yplatform/module_registration.h>
#include <yplatform/loader.h>

#include <yamail/data/deserialization/ptree_reader.h>

#include <mail/ymod_queuedb_worker/include/module.h>
#include <mail/ymod_queuedb_worker/include/task_handler.h>
#include <mail/ymod_queuedb_worker/include/task_control.h>
#include <mail/ymod_queuedb_worker/include/internal/error.h>
#include <mail/ymod_queuedb_worker/include/internal/timer.h>
#include <mail/ymod_queuedb_worker/include/internal/run_loop.h>
#include <mail/ymod_queuedb_worker/include/internal/as_seconds.h>

#include <mail/webmail/http_api_helpers/include/find_dependency.h>

#include <butil/network/utils.h>


namespace ymod_queuedb {

namespace ytt = yplatform::time_traits;

auto asMilliseconds(const yplatform::ptree& cfg, const std::string& key) {
    return ytt::duration_cast<std::chrono::milliseconds>(cfg.get<ytt::duration>(key));
}

Worker createWorker(size_t i) {
    std::ostringstream out;
    out << getHostName() << "_" << std::this_thread::get_id() << "_" << i;
    return Worker(out.str());
}

struct WorkerModuleImpl: public yplatform::module, public WorkerModule {
    TaskContextHolder running_;

    QueuePtr queuedb_;
    std::shared_ptr<yplatform::reactor> reactor_;
    std::string workerAccessLogName_;
    std::string logName_;
    unsigned coroutineStackSize_;
    std::chrono::milliseconds sleepTimeout_;
    bool logResponse_;
    TaskHandlersMap taskHandlers_;
    size_t numberOfCoroutines_;

    void readConfig(const yplatform::ptree& cfg) {
        using http_api::findDependency;

        queuedb_ = findDependency<Queue>(cfg, "dependencies.queuedb");
        reactor_ = findDependency<yplatform::reactor>(cfg, "dependencies.reactor");

        workerAccessLogName_ = cfg.get<std::string>("dependencies.worker_access_log");
        logName_ = cfg.get<std::string>("dependencies.log");
        coroutineStackSize_ = cfg.get<unsigned>("coroutine_stack_size");
        sleepTimeout_ = asMilliseconds(cfg, "sleep_timeout_dur");
        logResponse_ = cfg.get<bool>("log_response");
    }

    void start() {
        const auto accessLog = getWorkerAccessLogger(workerAccessLogName_);
        const auto logger = getLogger(logName_);
        for (size_t i = 0; i < numberOfCoroutines_; i++) {
            boost::asio::spawn(*reactor_->io(), [accessLog, logger, i, this] (boost::asio::yield_context yield) {
                Loop(
                    this->taskHandlers_, this->reactor_, this->queuedb_, this->sleepTimeout_,
                    accessLog, this->running_, logger, createWorker(i)
                ).run(yield);
            }, boost::coroutines::attributes(coroutineStackSize_));
        }
    }

    void stop() {
        running_.cancel();
    }

    void addHandler(TaskType type, Delay onFail, MaxRetries tries,
                    Delay onDelay, TaskHandler handler) override {
        if (taskHandlers_.find(type) != taskHandlers_.end()) {
            throw std::runtime_error(std::string("duplicate handler type: ") + static_cast<std::string>(type));
        }

        taskHandlers_[type] = TaskHandlerInfo {
            .maxRetries=tries, .onFail=onFail,
            .onDelay=onDelay, .handler=wrap(std::move(handler))
        };
    }

    TaskHandler wrap(TaskHandler fn) const {
        if (logResponse_) {
            return trycatchWithLog(getLogger(logName_), std::move(fn));
        } else {
            return trycatch(std::move(fn));
        }

    }
};

struct CoroutinePerThreadWorker: public WorkerModuleImpl {
    void init(const yplatform::ptree& cfg) {
        readConfig(cfg);
        numberOfCoroutines_ = reactor_->size();

        LOGDOG_(getLogger(logName_), notice,
                log::message="ymod_queuedb::CoroutinePerThreadWorker module loaded");
    }
};

struct SeveralCoroutinesPerThreadWorker: public WorkerModuleImpl {
    void init(const yplatform::ptree& cfg) {
        readConfig(cfg);
        numberOfCoroutines_ = cfg.get<size_t>("total_nubmer_of_coroutines");

        LOGDOG_(getLogger(logName_), notice,
                log::message="ymod_queuedb::SeveralCoroutinesPerThreadWorker module loaded");
    }
};

void addHandler(const yplatform::ptree& cfg,
                const TaskType& type,
                WorkerModule& worker,
                TaskHandler handler) {
    worker.addHandler(
        type,
        Delay(asSeconds(cfg, "on_fail_dur")),
        MaxRetries(cfg.get<uint32_t>("max_tries")),
        Delay(asSeconds(cfg, "on_delay_dur")),
        std::move(handler)
    );
}

}

DEFINE_SERVICE_OBJECT(ymod_queuedb::CoroutinePerThreadWorker)
DEFINE_SERVICE_OBJECT(ymod_queuedb::SeveralCoroutinesPerThreadWorker)
