#include <mail/ymod_queuedb_worker/include/internal/run_loop.h>
#include <mail/ymod_queuedb_worker/include/internal/error.h>
#include <mail/ymod_queuedb_worker/include/internal/timer.h>
#include <mail/ymod_queuedb_worker/include/internal/log.h>
#include <mail/ymod_queuedb_worker/include/task_control.h>

#include <yplatform/task_context.h>
#include <mail/ymod_queuedb/include/queue.h>
#include <mail/ymod_queuedb/include/logdog.h>
#include <boost/asio/spawn.hpp>
#include <yplatform/tskv/tskv.h>


namespace ymod_queuedb {

using namespace std::string_literals;

TaskHandler trycatch(TaskHandler fn) {
    return [fn=std::move(fn)] (const Task& task, bool lastTry, yplatform::task_context_ptr ctx, boost::asio::yield_context yield) -> yamail::expected<void> {
        try {
            return fn(task, lastTry, ctx, yield);
        } catch (const boost::coroutines::detail::forced_unwind&) {
            throw;
        } catch (const boost::system::system_error& e) {
            return yamail::make_unexpected(mail_errors::error_code(e.code()));
        } catch (const TaskControlDelayException&) {
            return yamail::make_unexpected(make_error(TaskControl::delay));
        } catch (const std::exception& e) {
            return yamail::make_unexpected(make_error(WorkerError::unexpectedException, e.what()));
        }
    };
}

TaskHandler trycatchWithLog(Logger logger, TaskHandler fn) {
    return [fn=std::move(fn), logger] (const Task& task, bool lastTry, yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
        const yamail::expected val = trycatch(fn)(task, lastTry, ctx, yield);

        if (val) {
            LOGDOG_(logger, debug,
                    log::task=task,
                    log::message="expected");
        } else {
            LOGDOG_(logger, error,
                    log::task=task,
                    log::message="unexpected",
                    log::error_code=val.error());
        }

        return val;
    };
}

Reason makeReason(const mail_errors::error_code& ec) {
    std::ostringstream out;
    out << ytskv::utf
        << ytskv::attr("category", ec.category().name())
        << ytskv::attr("message", ec.category().message(ec.value()))
        << ytskv::attr("reason", ec.what());

    return Reason(out.str());
}

Loop::Loop(TaskHandlersMap taskHandlers, std::shared_ptr<yplatform::reactor> reactor,
           QueuePtr queuedb, std::chrono::milliseconds sleepTimeout, WorkerAccessLogger accessLog,
           const TaskContextHolder& running, Logger logger, Worker worker)
    : taskHandlers(std::move(taskHandlers))
    , reactor(std::move(reactor))
    , queuedb(std::move(queuedb))
    , sleepTimeout(std::move(sleepTimeout))
    , accessLog(std::move(accessLog))
    , running(running)
    , logger(std::move(logger))
    , worker(std::move(worker))
{ }

std::function<void(boost::asio::yield_context)> Loop::createRefreshTaskCoro(std::weak_ptr<TaskId> weakTaskId,
                                                                            const TaskContextHolder& taskContextHolder,
                                                                            Timeout timeout, const std::string& requestId) const {
    Timer timer(reactor);
    auto ms = std::chrono::duration_cast<std::chrono::milliseconds>(
        static_cast<std::chrono::seconds>(timeout)
    );
    ms = ms / 4 * 3;

    return [=, logger=this->logger, queuedb=this->queuedb,
            worker=this->worker] (boost::asio::yield_context yield) {
        while (!taskContextHolder.cancelled()) {
            try {
                timer.asyncWait(ms, io_result::make_yield_context(yield));

                if (const auto taskId = weakTaskId.lock()) {
                    LOGDOG_(logger, debug,
                            log::where_name="runRefreshTaskCoro",
                            log::message="refreshing task with id: "s + std::to_string(*taskId),
                            log::request_id=requestId);
                    queuedb->refreshTask(
                        *taskId, worker, ymod_queuedb::RequestId(requestId),
                        io_result::make_yield_context(yield)
                    );
                } else {
                    LOGDOG_(logger, debug,
                            log::where_name="runRefreshTaskCoro",
                            log::message="stop refreshing task",
                            log::request_id=requestId);
                    taskContextHolder.cancel();
                }
            } catch (const boost::coroutines::detail::forced_unwind&) {
                throw;
            } catch (const boost::system::system_error& e) {
                if (e.code() == boost::asio::error::operation_aborted) {
                    LOGDOG_(logger, debug,
                            log::where_name="runRefreshTaskCoro",
                            log::message="stop refreshing task",
                            log::request_id=requestId);
                } else {
                    LOGDOG_(logger, error,
                            log::where_name="runRefreshTaskCoro",
                            log::request_id=requestId,
                            log::error_code=e.code());
                }
                taskContextHolder.cancel();
            } catch (const std::exception& e) {
                LOGDOG_(logger, error,
                        log::where_name="runRefreshTaskCoro",
                        log::request_id=requestId,
                        log::exception=e);
                taskContextHolder.cancel();
            }
        }
    };

}

void Loop::runRefreshTaskCoro(std::weak_ptr<TaskId> weakTaskId, const TaskContextHolder& taskContextHolder,
                              Timeout timeout, const std::string& requestId) const {
    boost::asio::spawn(*reactor->io(), createRefreshTaskCoro(weakTaskId, taskContextHolder, timeout, requestId));
}

void Loop::run(boost::asio::yield_context yield) const {
    while (!running.cancelled()) {
        std::optional<Task> task;
        TaskHandlersMap::const_iterator it = taskHandlers.end();
        TaskContextHolder taskContextHolder;
        try {
            task = acquireTasks(taskContextHolder, yield);

            if (task) {
                it = taskHandlers.find(task->task);

                LOGDOG_(logger, debug,
                        log::request_id=task->requestId,
                        log::task=*task,
                        log::message="acquired");

                if (it != taskHandlers.end()) {
                    processTask(*task, it->second, taskContextHolder, yield);
                } else {
                    failTask(*task, taskContextHolder, MAX_TRIES_FOR_ERROR, DELAY_FOR_ERROR,
                             make_error(WorkerError::unknownTaskType, task->task), yield);
                }
            } else if (!running.cancelled()) {
                Timer(reactor).asyncWait(sleepTimeout, io_result::make_yield_context(yield));
            }
        } catch (const boost::coroutines::detail::forced_unwind&) {
            throw;
        } catch (const boost::system::system_error& e) {
            LOGDOG_(logger, error,
                    log::where_name="runLoop",
                    log::request_id=task ? task->requestId : taskContextHolder.uniqId(),
                    log::exception=e);

            if (task && it != taskHandlers.end()) {
                failTask(*task, taskContextHolder, it->second.maxRetries, it->second.onFail,
                         make_error(WorkerError::unexpectedException, e.what()), yield, std::nothrow);
            }
        } catch (const std::exception& e) {
            LOGDOG_(logger, error,
                    log::where_name="runLoop",
                    log::request_id=task ? task->requestId : taskContextHolder.uniqId(),
                    log::exception=e);

            if (task && it != taskHandlers.end()) {
                failTask(*task, taskContextHolder, it->second.maxRetries, it->second.onFail,
                         make_error(WorkerError::unexpectedException, e.what()), yield, std::nothrow);
            }
        }
    }
}

std::optional<Task> Loop::acquireTasks(const TaskContextHolder& taskContextHolder, boost::asio::yield_context yield) const {
    mail_errors::error_code ec;
    TaskList tasks = queuedb->acquireTasks(
        worker, TasksLimit(1), ymod_queuedb::RequestId(taskContextHolder.uniqId()),
        io_result::make_yield_context(yield, ec)
    );

    if (ec) {
        LOGDOG_(logger, error,
                log::request_id=taskContextHolder.uniqId(),
                log::message="cannot acquire tasks",
                log::error_code=ec);

        return std::nullopt;
    } else if (tasks.empty()) {
        return std::nullopt;
    } else if (tasks.size() != 1) {
        throw std::runtime_error(
            "strange number of tasks: "s + std::to_string(tasks.size())
        );
    }

    return std::make_optional(std::move(tasks[0]));
}

void Loop::failTask(const Task& task, const TaskContextHolder& taskContextHolder,
                    MaxRetries maxRetries, Delay delay, const mail_errors::error_code& ec,
                    boost::asio::yield_context yield) const {
    queuedb->failTask(
        task.taskId, worker, makeReason(ec), maxRetries,
        delay, task.requestId,
        io_result::make_yield_context(yield)
    );
    accessLogFail(accessLog, *taskContextHolder.task, task);
}

void Loop::failTask(const Task& task, const TaskContextHolder& taskContextHolder,
                    MaxRetries maxRetries, Delay delay, const mail_errors::error_code& ec,
                    boost::asio::yield_context yield, const std::nothrow_t &) const {
    try {
        failTask(task, taskContextHolder, maxRetries, delay, ec, yield);
    } catch (const boost::coroutines::detail::forced_unwind&) {
        throw;
    } catch (const std::exception& e) {
        LOGDOG_(logger, error,
                log::message="cannot fail task",
                log::request_id=task.requestId,
                log::exception=e);
    } catch (...) {
        LOGDOG_(logger, error,
                log::message="cannot fail task with unknown exception",
                log::request_id=task.requestId);
    }
}

void Loop::completeTask(const Task& task, const TaskContextHolder& taskContextHolder,
                        boost::asio::yield_context yield) const {
    queuedb->completeTask(
        task.taskId, worker, task.requestId,
        io_result::make_yield_context(yield)
    );
    accessLogSuccess(accessLog, *taskContextHolder.task, task);
}

void Loop::delayTask(const Task& task, const TaskContextHolder& taskContextHolder, Delay delay,
                     boost::asio::yield_context yield) const {
    queuedb->delayTask(
        task.taskId, worker, delay, task.requestId,
        io_result::make_yield_context(yield)
    );
    accessLogDelayed(accessLog, *taskContextHolder.task, task);
}

void Loop::processTask(const Task& task, const TaskHandlerInfo& handler, const TaskContextHolder& contextHolder,
                       boost::asio::yield_context yield) const {
    const auto taskId = std::make_shared<TaskId>(task.taskId);
    runRefreshTaskCoro(taskId, contextHolder, task.timeoutSec, task.requestId);

    const bool lastTry = static_cast<std::uint32_t>(handler.maxRetries) <= task.tries+1;

    if (const auto resp = handler.handler(task, lastTry, contextHolder.task, yield)) {
        completeTask(task, contextHolder, yield);
    } else if (resp.error().category() == getTaskControlCategory()) {
        switch(static_cast<TaskControl>(resp.error().value())) {
            case TaskControl::delay:
                delayTask(task, contextHolder, handler.onDelay, yield);
            ; break;
            case TaskControl::permanentFail:
                failTask(task, contextHolder, MAX_TRIES_FOR_ERROR, DELAY_FOR_ERROR, resp.error(), yield);
            ; break;
        }
    } else {
        failTask(task, contextHolder, handler.maxRetries, handler.onFail, resp.error(), yield);
    }
}

}
