#pragma once

#include <variant>

#include <boost/range/adaptor/transformed.hpp>
#include <boost/range/algorithm/copy.hpp>

#include <mail/unistat/cpp/include/run.h>
#include <mail/unistat/cpp/include/meters/common.h>
#include <mail/unistat/cpp/include/common_logs.h>


namespace unistat {

struct WorkerTaskStat {
    WorkerTaskStat() = default;

    void update(const std::map<std::string, std::string>& record) {
        const auto taskType = record.find("task.type");
        const auto status = record.find("status");
        if (taskType == record.end() || status == record.end()) {
            return;
        }
        inc(taskType->second, status->second);
    }

    std::vector<unistat::NamedValue<std::size_t>> get() const {
        std::vector<unistat::NamedValue<std::size_t>> ret;
        ret.reserve(accum.size());

        boost::copy(accum
                    | boost::adaptors::transformed([](const auto& p) {
            return unistat::NamedValue<std::size_t>{unistat::withSigoptSuffix(p.first, "summ"), p.second};
        }), std::back_inserter(ret));

        return ret;
    }

private:
    static std::string fullName(const std::string& taskType, const std::string& name) {
        return "ctype=" + unistat::normalizeName(taskType) + ";" + name;
    }

    void inc(const std::string& taskType, const std::string& name) {
        const std::string full = fullName(taskType, name);
        if (accum.find(full) == accum.end()) {
            accum[full] = 1;
        } else {
            accum[full]++;
        }
    }

    std::map<std::string, std::size_t> accum;
};

using WorkerMeters = std::variant<std::shared_ptr<WorkerTaskStat>>;
using WorkerLog = unistat::Log<unistat::TextFileReader, unistat::File, unistat::TskvParser, WorkerMeters>;
using WorkerLogPtr = std::shared_ptr<WorkerLog>;

}
