#pragma once

#include <ymod_ratecontroller/errors.h>
#include <yplatform/time_traits.h>
#include <yplatform/task_context.h>

#include <boost/system/error_code.hpp>
#include <boost/asio/io_service.hpp>

#include <functional>
#include <memory>

namespace ymod_ratecontroller {

namespace time_traits = yplatform::time_traits;

using completion_handler = std::function<void()>;

struct rate_controller
{
    using task_type = std::function<void(error_code, completion_handler)>;

    virtual ~rate_controller() = default;

    virtual void post(
        const task_type& task,
        const std::string& task_id = "",
        const time_traits::time_point& deadline = time_traits::time_point::max()) = 0;
    virtual void cancel(const std::string& task_id) = 0;
    virtual std::size_t max_concurrency() const = 0;
    virtual std::size_t queue_size() const = 0;
    virtual std::size_t running_tasks_count() const = 0;

    virtual void post(yplatform::task_context_ptr ctx, const task_type& task)
    {
        validate_context(ctx);
        post(task, ctx.get()->uniq_id(), ctx.get()->deadline());
    }

    virtual void cancel(yplatform::task_context_ptr ctx)
    {
        validate_context(ctx);
        cancel(ctx.get()->uniq_id());
    }

private:
    void validate_context(yplatform::task_context_ptr ctx)
    {
        if (!ctx)
        {
            throw std::invalid_argument("task_context_ptr parameter is nullptr");
        }
    }
};

using rate_controller_ptr = std::shared_ptr<rate_controller>;

rate_controller_ptr create_controller(
    boost::asio::io_service& io,
    size_t max_concurrency,
    size_t max_queue_size);

struct rate_controller_module
{
    virtual ~rate_controller_module() = default;

    virtual rate_controller_ptr get_controller(const std::string& resource_path) = 0;
};

}
