#pragma once

#include "limit.h"
#include "config_reader.h"
#include "common.h"

#include <yaml-cpp/yaml.h>
#include <yplatform/ptree.h>

#include <format>
#include <memory>
#include <string>
#include <unordered_map>

namespace NYmodRateSrv {

class TClientHelperImpl {
public:
    void Init(const yplatform::ptree& configuration);

    TLimitPtr GetLimit(const std::string& limit);

    bool AddKeyToRequest(TRequest& request, const TLimit& limit, std::string id, ui64 value, TKey key);

private:
    void AddLimit(const std::string& limitName, const yplatform::ptree& limitPt, const std::string& groupName, const TGroupConfiguration& groupConfiguration);

private:
    std::unordered_map<std::string, TLimitPtr> Limits;
};

void TClientHelperImpl::Init(const yplatform::ptree& configuration) {
    const auto& limitsPt = configuration.get_child_optional("limits");
    if (!limitsPt) {
        return;
    }

    const std::string rulesFile = configuration.get<std::string>("rules_file", "");
    if (rulesFile.empty()) {
        throw std::invalid_argument("Rules configuration file is not specified");
    }

    const std::string groupName = configuration.get<std::string>("group_name");
    if (groupName.empty()) {
        throw std::invalid_argument("Group name must be not empty");
    }

    TGroupConfiguration groupConfiguration = ReadGroupConfiguration(rulesFile, groupName);

    for (const auto& [limitName, limitPt] : *limitsPt) {
        AddLimit(limitName, limitPt, groupName, groupConfiguration);
    }
}

void TClientHelperImpl::AddLimit(const std::string& limitName, const yplatform::ptree& limitPt, const std::string& groupName, const TGroupConfiguration& groupConfiguration) {
    if (Limits.count(limitName) > 0) {
        throw std::invalid_argument("Limit must be unique");
    }
    bool enable = limitPt.get<bool>("enable");
    bool dryRun = limitPt.get("dry_run", false);
    std::string rsLimitName;
    TLimitType limitType;
    TLimitConfiguration limitConfiguration;

    if (enable) {
        rsLimitName = limitPt.get<std::string>("limit_name");
        if (rsLimitName.empty()) {
            throw std::invalid_argument("Limit name must be not emtpy");
        }
        auto strType = limitPt.get<std::string>("type");
        limitType = GetLimitTypeFromString(strType);
        if (const auto limitConfigurationNode = groupConfiguration[rsLimitName]) {
            limitConfiguration = ReadLimitConfiguration(limitConfigurationNode);
        } else {
            throw std::invalid_argument(std::format("Group configuration of {} is missing limit called {}", groupName, rsLimitName));
        }
    }

    auto limit = std::make_shared<TLimit>(
        enable,
        dryRun,
        std::move(limitType),
        groupName,
        std::move(rsLimitName),
        std::move(limitConfiguration)
    );
    Limits.emplace(limitName, std::move(limit));
}

TLimitPtr TClientHelperImpl::GetLimit(const std::string& limit) {
    auto it = Limits.find(limit);
    if (it == Limits.end()) {
        return {};
    }
    return it->second;
}

bool TClientHelperImpl::AddKeyToRequest(
    TRequest& request,
    const TLimit& limit,
    std::string id,
    ui64 value,
    TKey key)
{
    auto part = limit.MakeRequestPart(value, std::move(key));
    return request.Add(std::move(id), std::move(part));
}

} // namespace NYmodRateSrv
