#include "common.h"

#include "key_parts/simple_string.h"
#include "key_parts/regexp_string.h"
#include "key_parts/ip.h"

#include <boost/tokenizer.hpp>

namespace NYmodRateSrv {

TLimitType GetLimitTypeFromString(const std::string& strType) {
    TLimitType limitType;

    boost::char_separator<char> sep("+");
    boost::tokenizer<boost::char_separator<char>> tokenizer(strType, sep);
    for (auto it = tokenizer.begin(); it != tokenizer.end(); ++it) {
        auto strPart = *it;
        if (strPart == "string") {
            limitType.push_back(EKeyPartType::String);
        } else if (strPart == "regex") {
            limitType.push_back(EKeyPartType::Regex);
        } else if (strPart == "ip") {
            limitType.push_back(EKeyPartType::Ip);
        } else {
            throw std::invalid_argument("Unknown part type " + strPart);
        }
    }

    if (limitType.empty()) {
        throw std::invalid_argument("Limit type must not be empty");
    }

    return limitType;
}

bool IsIncluded(const std::string& lhs, const std::string& rhs, EKeyPartType type) {
    switch (type) {
        case EKeyPartType::String:
            return NKeyParts::NSimpleString::IsIncluded(lhs, rhs);
        case EKeyPartType::Regex:
            return NKeyParts::NRegexpString::IsIncluded(lhs, rhs);
        case EKeyPartType::Ip:
            return NKeyParts::NIp::IsIncluded(lhs, rhs);
    }
}

TRuleNodePtr MakeRuleNodeByType(EKeyPartType type) {
    switch (type) {
        case EKeyPartType::String:
            return std::make_unique<NKeyParts::NSimpleString::TNode>();
        case EKeyPartType::Regex:
            return std::make_unique<NKeyParts::NRegexpString::TNode>();
        case EKeyPartType::Ip:
            return std::make_unique<NKeyParts::NIp::TNode>();
    }
}

std::string MakeKeyForType(const TKeyPart& part, EKeyPartType type) {
    switch (type) {
        case EKeyPartType::String:
            return NKeyParts::NSimpleString::MakeKey(part);
        case EKeyPartType::Regex:
            return NKeyParts::NRegexpString::MakeKey(part);
        case EKeyPartType::Ip:
            return NKeyParts::NIp::MakeKey(part);
    }
}

} // namespace NYmodRateSrv
