#include "config_unifier.h"

namespace NYmodRateSrv {

TConfigNode::TConfigNode(std::string value)
    : Value(std::move(value))
{}

void TConfigNode::Merge(TConfigNodePtr other) {
    for (auto& child : other->Childs) {
        Childs.push_back(std::move(child));
    }
}

namespace {

TConfigNodePtr MakeConfBranch(
    TConfigNodePtr node,
    const TRuleConf& ruleConf,
    const TLimitType& limitType,
    size_t domainId,
    size_t columnId)
{
    if (columnId >= ruleConf.size()) {
        node->DomainId = domainId;
        return node;
    }

    auto newNode = std::make_unique<TConfigNode>(ruleConf[columnId]);
    TConfigNodePtr* foundNode = nullptr;

    for (auto it = node->Childs.rbegin(); it != node->Childs.rend() && !foundNode;) {
        auto currIt = it++;
        auto& childNode = *currIt;

        if (IsIncluded(newNode->Value, childNode->Value, limitType[columnId])) {
            foundNode = &childNode;
            newNode.reset();
        } else if (IsIncluded(childNode->Value, newNode->Value, limitType[columnId])) {
            newNode->Merge(std::move(childNode));
            node->Childs.erase(std::prev(currIt.base()));
        }
    }

    if (!foundNode) {
        foundNode = &newNode;
    }
    *foundNode = MakeConfBranch(std::move(*foundNode), ruleConf, limitType, domainId, columnId + 1);

    if (newNode) {
        node->Childs.push_back(std::move(newNode));
    }

    return node;
}

} // namespace

TConfigNodePtr UnifyConfiguration(const TLimitConfiguration& configuration, const TLimitType& limitType) {
    auto root = std::make_unique<TConfigNode>();

    for (size_t i = 0; i < configuration.size(); ++i) {
        const auto& [domain, domainConf] = configuration[i];
        for (size_t j = 0; j < domainConf.size(); ++j) {
            const auto& ruleConf = domainConf[j];
            if (ruleConf.size() != limitType.size()) {
                throw std::runtime_error("Invalid column count in configuration domain " + domain +
                    " at row " + std::to_string(i) + ": expected " + std::to_string(limitType.size()) +
                    ", but " + std::to_string(ruleConf.size()));
            }
            root = MakeConfBranch(std::move(root), ruleConf, limitType, i, 0);
        }
    }

    return root;
}

} // namespace NYmodRateSrv
