#pragma once

#include <mail/ymod_ratesrv/helper/src/rule_node.h>

#include <boost/asio/ip/network_v4.hpp>
#include <boost/asio/ip/network_v6.hpp>
#include <boost/asio/ip/address.hpp>

#include <optional>
#include <string>
#include <utility>
#include <variant>
#include <vector>

/**
  * Key part for select domain by find the IP-address into network
 */

namespace NYmodRateSrv::NKeyParts::NIp {

using TAddress = boost::asio::ip::address;
using TNetwork4 = boost::asio::ip::network_v4;
using TNetwork6 = boost::asio::ip::network_v6;
using TNetwork = std::variant<TNetwork4, TNetwork6>;

std::string MakeKey(const TKeyPart& part);

bool IsIncluded(const std::string& lhs, const std::string& rhs);

TNetwork MakeNetworkFromString(const std::string& str);

template <typename TNetworkType>
bool Compare(const TNetwork& lhs, const TNetwork& rhs) {
    const auto& lNetwork = std::get<TNetworkType>(lhs);
    const auto& rNetwork = std::get<TNetworkType>(rhs);
    return lNetwork == rNetwork || lNetwork.is_subnet_of(rNetwork);
}

class TNode : public IRuleNode {
public:
    template <typename TNetwork>
    using TContainer = std::vector<std::pair<TNetwork, TChild>>;

    void Add(const std::string& value, TChild child) override {
        auto network = MakeNetworkFromString(value);
        if (!TryAddNetwork(network, child, Networks4)) {
            TryAddNetwork(network, child, Networks6);
        }
    }

    std::optional<size_t> Match(const TKey& key, size_t columnId) const override {
        const auto& part = key.at(columnId);
        if (!std::holds_alternative<TAddress>(part)) {
            throw std::invalid_argument("Invalid key part, expected ip at column " + std::to_string(columnId));
        }
        const auto& address = std::get<TAddress>(part);
        if (address.is_v4()) {
            return MatchByIp(TNetwork4(address.to_v4(), 32), Networks4, key, columnId);
        }
        return MatchByIp(TNetwork6(address.to_v6(), 128), Networks6, key, columnId);
    }

private:
    template <typename TNetworks>
    bool TryAddNetwork(TNetwork& network, TChild& child, TNetworks& networks) const {
        if (auto realNetwork = std::get_if<typename TNetworks::value_type::first_type>(&network)) {
            networks.emplace_back(std::move(*realNetwork), std::move(child));
            return true;
        }
        return false;
    }

    template <typename TNetworks>
    std::optional<size_t> MatchByIp(
        TNetwork network,
        TNetworks& networks,
        const TKey& key,
        size_t columnId) const
    {
        const auto& realNetwork = std::get<typename TNetworks::value_type::first_type>(network);
        for (const auto& [ruleNetwork, child] : networks) {
            if (realNetwork == ruleNetwork || realNetwork.is_subnet_of(ruleNetwork)) {
                if (auto domainId = std::get_if<size_t>(&child); domainId) {
                    return *domainId;
                }
                if (auto res = std::get<TRuleNodePtr>(child)->Match(key, columnId + 1); res) {
                    return res;
                }
            }
        }
        return {};
    }

private:
    TContainer<TNetwork4> Networks4;
    TContainer<TNetwork6> Networks6;
};

} // namespace NYmodRateSrv::NKeyParts::NIp
