#include "limit.h"

namespace NYmodRateSrv {

TLimit::TLimit(
    bool enabled,
    bool dryRun,
    TLimitType type,
    std::string groupName,
    std::string limitName,
    TLimitConfiguration configuration
)
    : Enabled(enabled)
    , DryRun(dryRun)
    , Type(std::move(type))
    , GroupName(std::move(groupName))
    , LimitName(std::move(limitName))
{
    Init(std::move(configuration));
}

void TLimit::Init(TLimitConfiguration configuration) {
    if (!Enabled) {
        return;
    } else if (Type.empty()) {
        throw std::logic_error("Type of limit must be not empty");
    }

    Domains.reserve(configuration.size());
    for (auto& [domain, domainConf] : configuration) {
        Domains.push_back(std::move(domain));
    }
    auto ruleConf = UnifyConfiguration(configuration, Type);
    Rules = MakeRuleTree(*ruleConf, 0);
}

TRuleNodePtr TLimit::MakeRuleTree(const TConfigNode& confNode, size_t level) {
    auto ruleNode = MakeRuleNodeByType(Type[level]);
    for (const auto& child : confNode.Childs) {
        if (level + 1 == Type.size()) {
            ruleNode->Add(child->Value, *child->DomainId);
        } else {
            ruleNode->Add(child->Value, MakeRuleTree(*child, level + 1));
        }
    }
    return ruleNode;
}

bool TLimit::IsEnabled() const {
    return Enabled;
}

bool TLimit::IsDryRun() const {
    return DryRun;
}

const TLimitType& TLimit::GetType() const {
    return Type;
}

std::string TLimit::GetGroupName() const {
    return GroupName;
}

std::string TLimit::GetLimitName() const {
    return LimitName;
}

TRequestPart TLimit::MakeRequestPart(ui64 value, TKey key) const {
    if (!Enabled) {
        throw std::runtime_error("Limit is disabled");
    }
    if (key.size() != Type.size()) {
        throw std::invalid_argument("Invalid arguments count, expected " + std::to_string(Type.size()) +
            ", but " + std::to_string(key.size()) + " provided");
    }
    std::string domain;
    if (auto domainId = Rules->Match(key, 0); domainId) {
        domain = Domains[*domainId];
    }
    std::string strKey;
    for (size_t i = 0; i < key.size(); ++i) {
        if (i > 0) {
            strKey += '#';
        }
        strKey += MakeKeyForType(key[i], Type[i]);
    }
    return {GetGroupName(), GetLimitName(), std::move(strKey), value, std::move(domain)};
}

} // namespace NYmodRateSrv
