#include <mail/ymod_ratesrv/helper/src/config_reader.h>

#include <library/cpp/resource/resource.h>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <sstream>

using namespace testing;
using namespace NYmodRateSrv;

MATCHER_P2(EqLimitConfig, expectedDomain, expectedDomainConf, "Limit configuration has expected domain and domain configuration") {
    const auto& [domain, domainConf] = arg;
    return domain == expectedDomain
        && domainConf == expectedDomainConf;
}

struct TTestConfigReader: Test {
    void Read(const std::string& file) {
        std::string strConfig = NResource::Find(file);
        std::istringstream stream(strConfig);
        Configuration = ReadLimitConfiguration(stream);
    }

    TLimitConfiguration Configuration;
};

TEST_F(TTestConfigReader, Single) {
    Read("string_limit.yml");
    ASSERT_THAT(Configuration, SizeIs(2));

    ASSERT_THAT(Configuration[0], EqLimitConfig("first_domain", TDomainConf {{"one"}, {"two"}, {"one"}, {"three"}}));
    ASSERT_THAT(Configuration[1], EqLimitConfig("second_domain", TDomainConf {{"four"}, {"five"}, {"six"}}));
}

TEST_F(TTestConfigReader, Double) {
    Read("string_and_ip_limit.yml");
    ASSERT_THAT(Configuration, SizeIs(2));

    ASSERT_THAT(Configuration[0], EqLimitConfig(
        "first_domain",
        TDomainConf {
            {"one", "127.0.0.0/8"},
            {"two", "192.168.0.0/16"},
            {"one", "127.255.0.0/16"},
            {"one", "10.10.10.0/24"},
            {"two", "192.0.0.0/8"},
            {"three", "127.0.0.1/24"},
        }
    ));

    ASSERT_THAT(Configuration[1], EqLimitConfig("second_domain", TDomainConf {{"one", "192.168.0.0/16"}, {"*", "172.16.0.0/16"}}));
}

struct TTestGroupConfigReader: Test {
    void Read(const std::string& file, const std::string& groupName) {
        std::string strConfig = NResource::Find(file);
        std::istringstream stream(strConfig);
        GroupConfiguration = ReadGroupConfiguration(stream, groupName);
        LimitConfigurations.reserve(GroupConfiguration.size());
        for (const auto limitNode: GroupConfiguration) {
            LimitConfigurations.push_back(ReadLimitConfiguration(limitNode.second));
        }
    }

    TGroupConfiguration GroupConfiguration;
    std::vector<TLimitConfiguration> LimitConfigurations;
};

TEST_F(TTestGroupConfigReader, First) {
    Read("group_configuration.yml", "first_group");
    ASSERT_THAT(GroupConfiguration, SizeIs(1));
    ASSERT_TRUE(GroupConfiguration["first_limit"]);

    ASSERT_THAT(LimitConfigurations.front()[0], EqLimitConfig(
        "first_domain",
        TDomainConf {
            {"one",   "127.0.0.0/8"   },
            {"two",   "192.168.0.0/16"},
            {"one",   "127.255.0.0/16"},
            {"one",   "10.10.10.0/24" },
            {"two",   "192.0.0.0/8"   },
            {"three", "127.0.0.1/24"  },
        }
    ));
}

TEST_F(TTestGroupConfigReader, Second) {
    Read("group_configuration.yml", "second_group");
    ASSERT_THAT(GroupConfiguration, SizeIs(2));
    ASSERT_TRUE(GroupConfiguration["first_limit"]);
    ASSERT_TRUE(GroupConfiguration["second_limit"]);
    ASSERT_THAT(LimitConfigurations[0], SizeIs(1));
    ASSERT_THAT(LimitConfigurations[1], SizeIs(1));

    ASSERT_THAT(LimitConfigurations[0][0], EqLimitConfig(
        "first_domain",
        TDomainConf {
            {"one",   "127.0.0.0/8"   },
            {"two",   "192.168.0.0/16"},
            {"one",   "127.255.0.0/16"},
            {"one",   "10.10.10.0/24" },
            {"two",   "192.0.0.0/8"   },
            {"three", "127.0.0.1/24"  },
        }
    ));

    ASSERT_THAT(LimitConfigurations[1][0], EqLimitConfig("second_domain", TDomainConf {{"one", "192.168.0.0/16"}, {"*", "172.16.0.0/16"}}));
}
