#include <mail/ymod_ratesrv/helper/src/limit.h>

#include <library/cpp/resource/resource.h>
#include <util/stream/output.h>

#include <boost/asio/ip/address.hpp>

#include <gmock/gmock.h>
#include <gtest/gtest.h>

#include <sstream>

using namespace testing;
using namespace NYmodRateSrv;
using boost::asio::ip::make_address;

template <>
void Out<TRequestPart>(IOutputStream& out, typename TTypeTraits<TRequestPart>::TFuncParam value) {
    out << "TRequestPart{" << value.Group << ":" << value.Limit;
    if (!value.Domain.empty()) {
        out << "@" << value.Domain;
    }
    out << ":" << value.Key << " -> " << value.Value;
}

namespace NYmodRateSrv {

bool operator==(const TRequestPart& lhs, const TRequestPart& rhs) {
    return
        lhs.Group == rhs.Group &&
        lhs.Limit == rhs.Limit &&
        lhs.Domain == rhs.Domain &&
        lhs.Key == rhs.Key &&
        lhs.Value == rhs.Value;
}

} // namespace NYmodRateSrv


struct TTestLimit: Test {
    using TAddress = boost::asio::ip::address;

    void ReadConfig(const std::string& file) {
        std::string strConfig = NResource::Find(file);
        std::istringstream stream(strConfig);
        Configuration = ReadLimitConfiguration(stream);
    }

    template <typename... TArgs>
    TRequestPart MakeRequestPartByLimit(const TLimit& limit, ui64 value, TArgs&&... args) {
        TKey parts{{std::forward<TArgs>(args)...}};
        return limit.MakeRequestPart(value, std::move(parts));
    }

    TLimitPtr CreateLimit(bool enable, bool dryRun, TLimitType type, std::string groupName, std::string limitName) {
        auto limit = std::make_shared<TLimit>(enable, dryRun, type, groupName, limitName, std::move(Configuration));

        EXPECT_EQ(limit->IsEnabled(), enable);
        EXPECT_EQ(limit->IsDryRun(), dryRun);
        EXPECT_TRUE(limit->GetType() == type);
        EXPECT_EQ(limit->GetGroupName(), groupName);
        EXPECT_EQ(limit->GetLimitName(), limitName);

        return limit;
    }

    TLimitConfiguration Configuration;
};

TEST_F(TTestLimit, String) {
    ReadConfig("string_limit.yml");
    auto limit = CreateLimit(true, false, {EKeyPartType::String}, "group", "limit");

    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 0, "ten"),
        (TRequestPart{"group", "limit", "ten", 0}));
    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 200, "one"),
        (TRequestPart{"group", "limit", "one", 200, "first_domain"}));
    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 100, "three"),
        (TRequestPart{"group", "limit", "three", 100, "first_domain"}));
    EXPECT_EQ(MakeRequestPartByLimit(
        *limit, 200, "four"),
        (TRequestPart{"group", "limit", "four", 200, "second_domain"}));
    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 300, "six"),
        (TRequestPart{"group", "limit", "six", 300, "second_domain"}));

    EXPECT_THROW(MakeRequestPartByLimit(*limit, 300, "one", "two"), std::invalid_argument);
}

TEST_F(TTestLimit, Ip) {
    ReadConfig("ip_limit.yml");
    auto limit = CreateLimit(true, true, {EKeyPartType::Ip}, "group", "limit");

    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 0, make_address("123.123.123.123")),
        (TRequestPart{"group", "limit", "123.123.123.123", 0}));
    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 200, make_address("192.168.1.1")),
        (TRequestPart{"group", "limit", "192.168.1.1", 200, "first_domain"}));
    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 100, make_address("2001:0DB8:000B:ABCD::1")),
        (TRequestPart{"group", "limit", "2001:db8:b:abcd::1", 100, "first_domain"}));
    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 200, make_address("127.120.11.0")),
        (TRequestPart{"group", "limit", "127.120.11.0", 200, "second_domain"}));
    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 300, make_address("127.100.0.1")),
        (TRequestPart{"group", "limit", "127.100.0.1", 300, "second_domain"}));

    EXPECT_THROW(MakeRequestPartByLimit(*limit, 300, make_address("192.168.1.1"), "two"), std::invalid_argument);
    EXPECT_THROW(MakeRequestPartByLimit(*limit, 300, "one"), std::invalid_argument);
}

TEST_F(TTestLimit, Regexp) {
    ReadConfig("regexp_limit.yml");
    auto limit = CreateLimit(true, true, {EKeyPartType::Regex}, "group", "limit");

    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 0, "ololo"),
        (TRequestPart{"group", "limit", "ololo", 0}));
    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 200, "simple"),
        (TRequestPart{"group", "limit", "simple", 200, "first_domain"}));
    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 100, "inbox@mail.ru"),
        (TRequestPart{"group", "limit", "inbox@mail.ru", 100, "first_domain"}));
    EXPECT_EQ(MakeRequestPartByLimit(
        *limit, 200, "TEST@GMAIL.COM"),
        (TRequestPart{"group", "limit", "TEST@GMAIL.COM", 200, "second_domain"}));
    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 300, "TEST@yandex.ru"),
        (TRequestPart{"group", "limit", "TEST@yandex.ru", 300, "second_domain"}));

    EXPECT_THROW(MakeRequestPartByLimit(*limit, 300, "simple", "hard"), std::invalid_argument);
}

TEST_F(TTestLimit, StringAndIp) {
    ReadConfig("string_and_ip_limit.yml");
    auto limit = CreateLimit(true, true, {EKeyPartType::String, EKeyPartType::Ip}, "group", "limit");

    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 0, "two", make_address("127.0.0.1")),
        (TRequestPart{"group", "limit", "two#127.0.0.1", 0}));
    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 200, "one", make_address("127.0.0.1")),
        (TRequestPart{"group", "limit", "one#127.0.0.1", 200, "first_domain"}));
    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 200, "two", make_address("192.168.1.1")),
        (TRequestPart{"group", "limit", "two#192.168.1.1", 200, "first_domain"}));
    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 200, "one", make_address("192.168.1.1")),
        (TRequestPart{"group", "limit", "one#192.168.1.1", 200, "second_domain"}));

    // Test of match by wildcard
    EXPECT_EQ(
        MakeRequestPartByLimit(*limit, 200, "five", make_address("172.16.1.1")),
        (TRequestPart{"group", "limit", "five#172.16.1.1", 200, "second_domain"}));

    EXPECT_THROW(MakeRequestPartByLimit(*limit, 300, "one", "two"), std::invalid_argument);
}
