#include <mail/ymod_ratesrv/helper/src/config_unifier.h>
#include <mail/ymod_ratesrv/helper/src/config_reader.h>

#include <library/cpp/resource/resource.h>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <sstream>
#include <tuple>
#include <type_traits>

using namespace testing;
using namespace NYmodRateSrv;

struct TTestMergeRules: Test {
    template <unsigned level>
    struct TMultiDimVector {
        using TType = std::vector<std::tuple<std::string, typename TMultiDimVector<level - 1>::TType>>;
    };

    template <>
    struct TMultiDimVector<0> {
        using TType = std::vector<std::tuple<std::string, size_t>>;
    };

    void Read(const std::string& file, TLimitType type) {
        std::string strConfig = NResource::Find(file);
        std::istringstream stream(strConfig);
        Rules = UnifyConfiguration(ReadLimitConfiguration(stream), type);
    }

    template <unsigned level>
    void CompareRules(const TConfigNode& node, const typename TMultiDimVector<level>::TType& rules) {
        ASSERT_EQ(node.Childs.size(), rules.size());
        for (size_t i = 0; i < rules.size(); ++i) {
            const auto& childNode = *node.Childs[i];
            EXPECT_EQ(childNode.Value, std::get<0>(rules[i]));
            if constexpr (level > 0) {
                CompareRules<level - 1>(childNode, std::get<1>(rules[i]));
            } else {
                ASSERT_TRUE(childNode.DomainId.has_value());
                EXPECT_EQ(*childNode.DomainId, std::get<1>(rules[i]));
            }
        }
    }

    template <unsigned level>
    void Compare(
        const TConfigNode& node,
        const typename TMultiDimVector<level>::TType& expectedRules)
    {
        CompareRules<level>(node, expectedRules);
    }

    TConfigNodePtr Rules;
};

TEST_F(TTestMergeRules, String) {
    Read("string_limit.yml", {EKeyPartType::String});

    Compare<0>(
        *Rules,
        {
            {"one", 0},
            {"two", 0},
            {"three", 0},
            {"four", 1},
            {"five", 1},
            {"six", 1}
        });
}

TEST_F(TTestMergeRules, Regexp) {
    Read("regexp_limit.yml", {EKeyPartType::Regex});

    Compare<0>(
        *Rules,
        {
            {"simple", 0},
            {".*@mail.ru", 0},
            {".*@gmail.com", 1},
            {"test@.*", 1}
        });
}

TEST_F(TTestMergeRules, Ip) {
    Read("ip_limit.yml", {EKeyPartType::Ip});

    Compare<0>(
        *Rules,
        {
            {"192.168.0.0/16", 0},
            {"10.0.0.0/8", 0},
            {"2001:0DB8:000B::/48", 0},
            {"2001:0DB8:0:ABCD::1234/128", 0},
            {"127.0.0.0/8", 1}
        });
}

TEST_F(TTestMergeRules, StringAndIp) {
    Read("string_and_ip_limit.yml", {EKeyPartType::String, EKeyPartType::Ip});

    Compare<1>(
        *Rules,
        {
            {"one", {{"127.0.0.0/8", 0}, {"10.10.10.0/24", 0}, {"192.168.0.0/16", 1}}},
            {"two", {{"192.0.0.0/8", 0}}},
            {"three", {{"127.0.0.1/24", 0}}},
            {"*", {{"172.16.0.0/16", 1}}}
        });
}

TEST_F(TTestMergeRules, IpInvalid) {
    EXPECT_THROW(Read("string_limit.yml", {EKeyPartType::Ip}), std::invalid_argument);
}

TEST_F(TTestMergeRules, StringAndIpInvalid) {
    EXPECT_THROW(
        (Read("string_and_ip_limit_invalid.yml", {EKeyPartType::String, EKeyPartType::Ip})),
        std::runtime_error);
}
