#include "request.h"
#include "util.h"

#include <library/cpp/json/json_writer.h>

#include <util/stream/str.h>
#include <util/random/easy.h>

#include <algorithm>

namespace NYmodRateSrv {

TRequestPart::TRequestPart(
    std::string group,
    std::string limit,
    std::string key,
    ui64 value,
    std::string domain
)
    : Group(std::move(group))
    , Limit(std::move(limit))
    , Key(std::move(key))
    , Domain(std::move(domain))
    , Value(value)
{}

TRequestPart::TRequestPart(
    std::string group,
    std::string limit,
    std::string key,
    std::string domain
)
    : TRequestPart(std::move(group), std::move(limit), std::move(key), 0, std::move(domain))
{}

bool TRequest::Add(std::string id, TRequestPart part) {
    CheckCounter(id, part.Group, part.Limit, part.Key, part.Domain);
    return Request.emplace(std::move(id), std::move(part)).second;
}

std::string TRequest::Add(TRequestPart part) {
    std::string id;

    do {
        id = std::to_string(static_cast<unsigned int>(Random()));
    } while (!Add(id, part));

    return id;
}

size_t TRequest::Size() const {
    return Request.size();
}

void TRequest::Clear() {
    Request.clear();
}

bool TRequest::HasPartWithDomain(const std::string& domain) const {
    return std::find_if(Request.begin(), Request.end(), [&](const auto& node) {
               return node.second.Domain == domain;
           })
        != Request.end();
}

std::string TRequest::MakeGetRequest() const {
    TStringStream stream;
    NJson::TJsonWriter writer(&stream, false);

    writer.OpenMap();
    writer.Write("counters");
    writer.OpenMap();

    for (const auto& [id, part] : Request) {
        writer.Write(id);
        writer.Write(MakeCounter(part.Group, part.Limit, part.Key, part.Domain));
    }

    writer.CloseMap();
    writer.CloseMap();
    writer.Flush();

    return static_cast<std::string>(stream.Str());
}

std::string TRequest::MakeIncreaseRequest() const {
    TStringStream stream;
    NJson::TJsonWriter writer(&stream, false);

    writer.OpenMap();
    writer.Write("counters");
    writer.OpenMap();

    for (const auto& [id, part] : Request) {
        if (part.Value == 0) {
            throw std::invalid_argument("Increase values must be greater than 0");
        }

        writer.Write(id);

        writer.OpenMap();
        writer.Write("name");
        writer.Write(MakeCounter(part.Group, part.Limit, part.Key, part.Domain));
        writer.Write("value");
        writer.Write(part.Value);
        writer.CloseMap();
    }

    writer.CloseMap();
    writer.CloseMap();
    writer.Flush();

    return static_cast<std::string>(stream.Str());
}

} // namespace NYmodRateSrv
