#include "worker.h"
#include "errors.h"

#include <yplatform/yield.h>

#include <sstream>

namespace NYmodRateSrv {

namespace {

const char GetUrl[] = "/counters";
const char IncreaseUrl[] = "/counters/increase";

} // namespace

TWorker::TWorker(
    THttpClientPtr httpClient,
    TTaskContextPtr ctx,
    EMode mode,
    const TRequest& request,
    TCallback callback
)
    : HttpClient(std::move(httpClient))
    , Ctx(std::move(ctx))
    , Mode(mode)
    , Request(Mode == EMode::Increase ? request.MakeIncreaseRequest() : request.MakeGetRequest())
    , Size(request.Size())
    , Callback(std::move(callback))
{}

void TWorker::operator()(TYieldCtx yieldCtx, boost::system::error_code ec, yhttp::response response) {
    TResponse res;

    reenter (yieldCtx) {
        StartedAt = yplatform::time_traits::clock::now();

        yield HttpClient->async_run(
            Ctx,
            yhttp::request::POST(Mode == EMode::Get ? GetUrl : IncreaseUrl, std::move(Request)),
            yieldCtx);

        if (!ec) {
            try {
                std::tie(ec, res) = ParseHttpResponse(response);
            } catch (const std::exception& exp) {
                ec = EC_PARSE_ERROR;
                YLOG_CTX_LOCAL(Ctx, error) << "fail to parse response, http_code="
                    << response.status << ", error: " << exp.what();
            }
        }

        WriteLog(ec);
        Callback(std::move(ec), std::move(res));
    }
}

std::pair<boost::system::error_code, TResponse> TWorker::ParseHttpResponse(const yhttp::response& response) {
    if (response.status == 200) {
        return {{}, MakeResponseFromJson(response.body)};
    }

    return {MakeErrorFromHttpStatus(response.status), {}};
}

void TWorker::WriteLog(const boost::system::error_code& ec) {
    std::stringstream ss;
    ss << "request size=" << Size
       << ", delay=" << yplatform::time_traits::to_string(yplatform::time_traits::clock::now() - StartedAt)
       << ", status=";

    if (ec) {
        ss << "error (" << ec.message() << ")";
        YLOG_CTX_LOCAL(Ctx, error) << ss.str();
    } else {
        ss << "ok";
        YLOG_CTX_LOCAL(Ctx, info) << ss.str();
    }
}

} // namespace NYmodRateSrv
