#include <mail/ymod_ratesrv/src/request.h>

#include <library/cpp/json/json_reader.h>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

using namespace testing;
using namespace NYmodRateSrv;

namespace {

NJson::TJsonValue::TMapType MakeJsonMap(const std::string& json) {
    NJson::TJsonValue res;
    NJson::ReadJsonTree(json, &res, true);
    return res.GetMap();
}

} // namespace

TEST(TTestRequest, Empty) {
    TRequest request;
    EXPECT_EQ(request.Size(), 0u);
    EXPECT_EQ(MakeJsonMap(request.MakeGetRequest()), MakeJsonMap("{\"counters\": {}}"));
    EXPECT_EQ(MakeJsonMap(request.MakeIncreaseRequest()), MakeJsonMap("{\"counters\": {}}"));
}

TEST(TTestRequest, Get) {
    TRequest request;
    EXPECT_TRUE(request.Add("id1", TRequestPart("group", "limit", "key1")));
    EXPECT_TRUE(request.Add("id2", TRequestPart("group", "limit", "key2", "domain")));
    EXPECT_TRUE(request.Add("id3", TRequestPart("group", "limit", "key3", 100))); // value must be ignored
    auto genId = request.Add(TRequestPart("group", "unlimit", "key"));
    EXPECT_EQ(request.Size(), 4u);

    const std::string json = R"({"counters": {
        "id1": "group:limit:key1",
        "id2": "group:limit@domain:key2",
        "id3": "group:limit:key3",
        ")" + genId + R"(": "group:unlimit:key"
    }})";

    EXPECT_EQ(MakeJsonMap(request.MakeGetRequest()), MakeJsonMap(json));
}

TEST(TTestRequest, Increase) {
    TRequest request;
    EXPECT_TRUE(request.Add("id1", TRequestPart("group", "limit", "key1", 100)));
    EXPECT_TRUE(request.Add("id2", TRequestPart("group", "limit", "key2", 200, "domain")));
    EXPECT_TRUE(request.Add("id3", TRequestPart("group", "limit", "key3", 300)));
    auto genId = request.Add(TRequestPart("group", "unlimit", "key", 400));
    EXPECT_EQ(request.Size(), 4u);

    const std::string json = R"({"counters": {
        "id1": {"name": "group:limit:key1", "value": 100},
        "id2": {"name": "group:limit@domain:key2", "value": 200},
        "id3": {"name": "group:limit:key3", "value": 300},
        ")" + genId + R"(": {"name": "group:unlimit:key", "value": 400}
    }})";

    EXPECT_EQ(MakeJsonMap(request.MakeIncreaseRequest()), MakeJsonMap(json));
}

TEST(TTestRequest, IncreaseZero) {
    TRequest request;
    EXPECT_TRUE(request.Add("id1", TRequestPart("group", "limit", "key1", 100)));
    EXPECT_TRUE(request.Add("id2", TRequestPart("group", "limit", "key2", 0)));

    EXPECT_THROW(request.MakeIncreaseRequest(), std::invalid_argument);
}

TEST(TTestRequest, DuplicationAndClear) {
    TRequest request;
    request.Add("id1", TRequestPart("group", "limit", "key1"));
    request.Add("id2", TRequestPart("group", "limit", "key2", "domain"));
    EXPECT_FALSE(request.Add("id1", TRequestPart("unknown", "unknown", "key")));
    EXPECT_EQ(request.Size(), 2u);

    request.Clear();
    EXPECT_EQ(request.Size(), 0u);
}

TEST(TTestRequest, HasPartWithDomain) {
    TRequest request;
    request.Add("id1", TRequestPart("group", "limit", "key1"));
    request.Add("id2", TRequestPart("group", "limit", "key2", "domain"));
    request.Add("id2", TRequestPart("group", "limit", "key2", "domain2"));
    EXPECT_FALSE(request.HasPartWithDomain("not_domain"));
    EXPECT_TRUE(request.HasPartWithDomain("domain"));
}
