#include "print_counter.h"

#include <mail/ymod_ratesrv/src/worker.h>
#include <mail/ymod_ratesrv/src/errors.h>

#include <yplatform/coroutine.h>

#include <library/cpp/json/json_reader.h>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <functional>

using namespace testing;
using namespace NYmodRateSrv;

namespace {

class THttpClient : public ymod_httpclient::cluster_call {
public:
    using THttpCallback = std::function<void(yhttp::request req, callback_type callback)>;

    THttpClient(THttpCallback httpCallback)
        : HttpCallback(std::move(httpCallback))
    {}

    void async_run(task_context_ptr, yhttp::request req, callback_type callback) override {
        ++CallCount;
        HttpCallback(std::move(req), std::move(callback));
    }

    void async_run(task_context_ptr ctx, yhttp::request req, const options&, callback_type callback) override {
        async_run(std::move(ctx), std::move(req), std::move(callback));
    }

public:
    THttpCallback HttpCallback;
    size_t CallCount = 0;
};

struct TTestWorker: Test {
    void Run(
        THttpClient::THttpCallback httpCallback,
        TWorker::EMode mode,
        const TRequest& request,
        TWorker::TCallback callback
    ) {
        auto httpClient = std::make_shared<THttpClient>(std::move(httpCallback));
        yplatform::task_context_ptr ctx(new yplatform::task_context());

        auto worker = std::make_shared<TWorker>(httpClient, ctx, mode, request, std::move(callback));
        yplatform::spawn(worker);

        EXPECT_EQ(httpClient->CallCount, 1u);
    }
};

NJson::TJsonValue::TMapType MakeJsonMap(const std::string& json) {
    NJson::TJsonValue res;
    NJson::ReadJsonTree(json, &res, true);
    return res.GetMap();
}

} // namespace

TEST_F(TTestWorker, Get) {
    TRequest request;
    request.Add("id1", TRequestPart("group", "limit", "key1"));
    request.Add("id2", TRequestPart("group", "limit", "key2", "domain"));

    const std::string json = R"({"counters": {
        "id1": "group:limit:key1",
        "id2": "group:limit@domain:key2"
    }})";

    auto httpCallback = [json](yhttp::request req, ymod_httpclient::cluster_call::callback_type callback) {
        EXPECT_TRUE(req.method == yhttp::request::method_t::POST);
        EXPECT_EQ(req.url, "/counters");
        EXPECT_EQ(MakeJsonMap(*req.body), MakeJsonMap(json));

        yhttp::response response;
        response.status = 200;
        response.body = R"({"counters": {
            "id1": {"status": "ok", "current": 100, "available": 200},
            "id2": {"status": "exceeded", "current": 200, "available": -100}
        }})";

        callback({}, std::move(response));
    };

    bool callbackFinished = false;
    auto clientCallback = [&callbackFinished](boost::system::error_code ec, TResponse response) {
        EXPECT_TRUE(!ec);
        EXPECT_THAT(response, SizeIs(2));
        EXPECT_EQ(response["id1"], (TCounter{ECounterState::Ok, 100, 200, ""}));
        EXPECT_EQ(response["id2"], (TCounter{ECounterState::Exceeded, 200, -100, ""}));
        callbackFinished = true;
    };

    Run(std::move(httpCallback), TWorker::EMode::Get, request, std::move(clientCallback));
    EXPECT_TRUE(callbackFinished);
}

TEST_F(TTestWorker, Increase) {
    TRequest request;
    EXPECT_TRUE(request.Add("id1", TRequestPart("group", "limit", "key1", 100)));
    EXPECT_TRUE(request.Add("id2", TRequestPart("group", "limit", "key2", 200, "domain")));

    const std::string json = R"({"counters": {
        "id1": {"name": "group:limit:key1", "value": 100},
        "id2": {"name": "group:limit@domain:key2", "value": 200}
    }})";

    auto httpCallback = [json](yhttp::request req, ymod_httpclient::cluster_call::callback_type callback) {
        EXPECT_TRUE(req.method == yhttp::request::method_t::POST);
        EXPECT_EQ(req.url, "/counters/increase");
        EXPECT_EQ(MakeJsonMap(*req.body), MakeJsonMap(json));

        yhttp::response response;
        response.status = 200;
        response.body = R"({"counters": {
            "id1": {"status": "ok", "current": 100, "available": 200},
            "id2": {"status": "exceeded", "current": 200, "available": -100}
        }})";

        callback({}, std::move(response));
    };

    bool callbackFinished = false;
    auto clientCallback = [&callbackFinished](boost::system::error_code ec, TResponse response) {
        EXPECT_TRUE(!ec);
        EXPECT_THAT(response, SizeIs(2));
        EXPECT_EQ(response["id1"], (TCounter{ECounterState::Ok, 100, 200, ""}));
        EXPECT_EQ(response["id2"], (TCounter{ECounterState::Exceeded, 200, -100, ""}));
        callbackFinished = true;
    };

    Run(std::move(httpCallback), TWorker::EMode::Increase, request, std::move(clientCallback));
    EXPECT_TRUE(callbackFinished);
}

TEST_F(TTestWorker, ParseError) {
    auto httpCallback = [](yhttp::request, ymod_httpclient::cluster_call::callback_type callback) {
        yhttp::response response;
        response.status = 200;
        response.body = "ololo";

        callback({}, std::move(response));
    };

    bool callbackFinished = false;
    auto clientCallback = [&callbackFinished](boost::system::error_code ec, TResponse) {
        EXPECT_TRUE(ec == EErrorCode::EC_PARSE_ERROR);
        callbackFinished = true;
    };

    Run(std::move(httpCallback), TWorker::EMode::Get, {}, std::move(clientCallback));
    EXPECT_TRUE(callbackFinished);
}

TEST_F(TTestWorker, NetworkError) {
    auto httpCallback = [](yhttp::request, ymod_httpclient::cluster_call::callback_type callback) {
        callback(boost::asio::error::basic_errors::connection_refused, {});
    };

    bool callbackFinished = false;
    auto clientCallback = [&callbackFinished](boost::system::error_code ec, TResponse) {
        EXPECT_TRUE(ec == boost::system::errc::connection_refused);
        callbackFinished = true;
    };

    Run(std::move(httpCallback), TWorker::EMode::Get, {}, std::move(clientCallback));
    EXPECT_TRUE(callbackFinished);
}

TEST_F(TTestWorker, HttpError) {
    auto httpCallback = [](yhttp::request, ymod_httpclient::cluster_call::callback_type callback) {
        yhttp::response response;
        response.status = 500;

        callback({}, std::move(response));
    };

    bool callbackFinished = false;
    auto clientCallback = [&callbackFinished](boost::system::error_code ec, TResponse) {
        EXPECT_TRUE(ec == EErrorCode::EC_SERVER_ERROR);
        callbackFinished = true;
    };

    Run(std::move(httpCallback), TWorker::EMode::Get, {}, std::move(clientCallback));
    EXPECT_TRUE(callbackFinished);
}
