#pragma once

#include <boost/system/error_code.hpp>
#include <cinttypes>
#include <vector>

namespace ymod_smtpclient {
namespace error {

enum Code : std::uint8_t {
    Success     = 0,
    PartialSend,
    TaskCancelled,
    ConnectionExpired,
    ConnectionClosedError,
    ConnectError,
    SslError,
    ReadError,
    WriteError,
    ParseResponseError,
    AuthError,
    AuthWithoutSSL,
    ProtocolError,
    ConnectionTimedOut,
    RequestTimedOut,
    BadRecipient,
    BadReplyCode,
    Unknown,

    COUNT
};

inline const std::string& message(Code code) {
    static constexpr std::initializer_list<const char*> names = {
        "Success",
        "Partial send message",
        "Task cancelled",
        "Connection expired",
        "Connection closed",
        "Connect error",
        "Ssl handshake error",
        "Read error",
        "Write error",
        "Parse response error",
        "Auth error",
        "Trying to authorize through unsecure channel",
        "Protocol error",
        "Connection timed out",
        "Request timed out",
        "Bad recipient",
        "Bad reply code from server",
        "Unknown error"
    };
    static_assert(names.size() == Code::COUNT, "Error codes count doesn't correspond with error names count");
    static const std::vector<std::string> errNames(names.begin(), names.end());
    return errNames[code];
}

class ErrorCategory: public boost::system::error_category {
public:
    const char* name() const noexcept { return "ymod_smtpclient"; }

    std::string message(int ev) const { return error::message(Code(ev)); }

    static const ErrorCategory& instance() {
        static const ErrorCategory category;
        return category;
    }
};

inline boost::system::error_code make_error_code(Code code) {
    return boost::system::error_code(static_cast<int>(code), ErrorCategory::instance());
}

}   // namespace error
}   // namespace ymod_smtpclient

namespace boost {
namespace system {

template <>
struct is_error_code_enum<ymod_smtpclient::error::Code>: std::true_type {
};

}   // namespace system
}   // namespace boost
