#pragma once

#include <ymod_smtpclient/auth.h>
#include <ymod_smtpclient/smtp_point.h>
#include <yplatform/time_traits.h>

#include <boost/optional.hpp>

namespace ymod_smtpclient {

template <typename T>
using Optional = boost::optional<T>;

struct Timeouts {
    using duration = yplatform::time_traits::duration;

    Timeouts() = default;
    Timeouts(
        duration connect,
        duration connectAttempt,
        duration command,
        duration data
    )
        : connect(connect)
        , connectAttempt(connectAttempt)
        , command(command)
        , data(data)
    {}

    duration connect = duration::max();
    duration connectAttempt = duration::max();
    duration command = duration::max();
    duration data = duration::max();
};

struct Options {
    Timeouts timeouts;
    Optional<bool> reuseSession;
    Optional<bool> dotStuffing;
    Optional<bool> allowRcptToErrors;
    Optional<bool> useSsl;
};

struct MailFrom {
    MailFrom(const std::string& email = ""): email(email) {}

    std::string email;
    Optional<std::string> envid;    // ENVID=<envid>
};

enum class NotifyMode {
    Never,
    Success,
    Failure,
    Delay
};

std::string to_string(NotifyMode type);

struct RcptTo {
    RcptTo(const std::string& email = ""): email(email) {}

    std::string email;
    std::vector<NotifyMode> notifyModes;    // NOTIFY=<mode1>,...,<modeN>
};

class RequestBuilder;

struct Request {
    SmtpPoint address;
    MailFrom mailfrom;
    std::vector<RcptTo> rcpts;
    Optional<AuthData> auth;
    std::shared_ptr<std::string> message;
};

class RequestBuilder {
public:
    RequestBuilder& address(const SmtpPoint& point);
    RequestBuilder& mailfrom(MailFrom value);
    RequestBuilder& auth(AuthData authData);
    RequestBuilder& addRcpt(RcptTo rcpt);

    template <typename Iterator>
    RequestBuilder& addRcpts(Iterator first, Iterator last) {
        for (auto email = first; email != last; ++email) {
            data.rcpts.emplace_back(*email);
        }
        return *this;
    }

    RequestBuilder& message(std::string msg);
    Request release();

private:
    Request data;
};

}   // namespace ymod_smtpclient
