#pragma once

#include <boost/optional.hpp>
#include <ostream>
#include <map>
#include <string>
#include <vector>
#include <cinttypes>

namespace ymod_smtpclient {

template <typename T>
using Optional = boost::optional<T>;

struct EnhancedStatusCode {
    enum ClassType: std::uint8_t {
        Success                     = 2,
        PersistentTransientFailure  = 4,
        PermanentFailure            = 5,
        Unknown
    };

    enum SubjectType: std::uint16_t {
        Undefined               = 0,
        Addressing              = 1,
        Mailbox                 = 2,
        MailSystem              = 3,
        NetworkAndRouting       = 4,
        MailDeliveryProtocol    = 5,
        MessageContent          = 6,
        Security                = 7
    };

    EnhancedStatusCode() = default;
    EnhancedStatusCode(ClassType c, SubjectType s, std::uint16_t d)
        : classType(c)
        , subjectType(s)
        , detail(d)
    {}

    ClassType classType = Unknown;
    SubjectType subjectType = Undefined;
    std::uint16_t detail = 0;
};

inline std::ostream& operator<<(std::ostream& stream, const EnhancedStatusCode& enhanced) {
    stream << static_cast<unsigned>(enhanced.classType) 
           << "." << static_cast<unsigned>(enhanced.subjectType) 
           << "." << enhanced.detail;
    return stream;
}

struct OneResponse {
    uint16_t replyCode = 0;
    Optional<EnhancedStatusCode> enhancedStatusCode;
    std::string data;
};

inline std::ostream& operator<<(std::ostream& stream, const OneResponse& response) {
    if (response.replyCode) {
        stream << response.replyCode << " ";
    }
    if (response.enhancedStatusCode) {
        stream << *response.enhancedStatusCode << " ";
    }
    stream << response.data;
    return stream;
}

struct Response {
    std::map<std::string, OneResponse> rcpts;    // email -> single response
    boost::optional<OneResponse> session;
};

struct MultiLineResponse {
    uint16_t replyCode = 0;
    Optional<EnhancedStatusCode> enhancedCode;
    std::vector<std::string> dataLines;
};

namespace reply {

enum class Status {
    Accept = 0,
    Reject,
    Tempfail,
    Unknown
};

inline Status smtp_code_to_status(uint16_t replyCode) {
    if (replyCode >= 200 && replyCode < 300) {
        return Status::Accept;
    } else if (replyCode >= 400 && replyCode < 500) {
        return Status::Tempfail;
    } else if (replyCode >= 500 && replyCode < 600) {
        return Status::Reject;
    }
    return Status::Unknown;
}

inline std::string to_string(Status status) {
    if (status == Status::Accept) {
        return "accept";
    } else if (status == Status::Reject) {
        return "reject";
    } else if (status == Status::Tempfail) {
        return "tempfail";
    }
    return "unknown";
}

}   // namespace reply

}   // namespace ymod_smtpclient
