#include <ymod_smtpclient/client.h>

#include "session.h"
#include "session_info.h"
#include "session_pool.h"

#include <deque>
#include <functional>

namespace ymod_smtpclient {

namespace p = std::placeholders;

class Client::Impl
    : public std::enable_shared_from_this<Client::Impl>
    , public yplatform::log::contains_logger
    , public boost::noncopyable
{
public:
    using Duration = yplatform::time_traits::duration;
    using SessionPoolPtr = std::shared_ptr<SessionPool>;
    using Connections = std::map<SessionInfo, SessionPoolPtr>;

    Impl(boost::asio::io_service& ios,
        const yplatform::log::source& logger,
        const Settings& settings
    )
        : yplatform::log::contains_logger(logger)
        , iodata(ios)
        , settings(settings)
    {
        iodata.setup_ssl(settings.ssl);
        iodata.setup_dns(settings.dns);
    }

    void asyncRun(Context ctx, Request request, Options opts, Callback callback) {
        auto reqData = std::make_shared<RequestData>();
        reqData->ctx = std::move(ctx);
        reqData->data = std::move(request);
        reqData->callback = std::move(callback);
        setOptions(reqData, opts);

        pendingRequests.push_back(std::move(reqData));
        processNextRequest();
    }

    SmtpSessionPtr createSmtpSession(Context ctx) {
        auto session = std::make_shared<SmtpSessionImpl>(iodata, settings, logger());
        session->setContext(ctx);
        return session;
    }

    SmtpSessionPtr createSmtpSession(Context ctx, const Timeouts& timeouts) {
        auto session = std::make_shared<SmtpSessionImpl>(iodata, settings, logger());
        session->setTimeouts(timeouts);
        session->setContext(ctx);
        return session;
    }

    boost::asio::io_service* get_io_service() { return iodata.get_io(); }

private:
    void setOptions(RequestDataPtr& request, const Options& opts) {
        request->timeouts.connect = (opts.timeouts.connect == Duration::max())
            ? settings.connectTimeout
            : opts.timeouts.connect;
        request->timeouts.connectAttempt = settings.connectAttemptTimeout;
        request->timeouts.command = (opts.timeouts.command == Duration::max())
            ? settings.commandTimeout
            : opts.timeouts.command;
        request->timeouts.data = (opts.timeouts.data == Duration::max())
            ? settings.dataTimeout
            : opts.timeouts.data;
        request->reuseConnection = opts.reuseSession ? *opts.reuseSession : settings.reuseConnection;
        request->dotStuffing = opts.dotStuffing ? *opts.dotStuffing : settings.dotStuffing;
        request->allowRcptToErrors = opts.allowRcptToErrors
            ? *opts.allowRcptToErrors
            : settings.allowErrorsOnRcptTo;
        request->useSsl = opts.useSsl;
    }

    void processNextRequest() {
        if (pendingRequests.empty()) {
            return;
        }
        auto request = std::move(pendingRequests.front());
        pendingRequests.pop_front();
        if (request->ctx->is_cancelled()) {
            get_io_service()->post([request] {
                request->callback(error::Code::TaskCancelled, Response{});
            });
            get_io_service()->post(std::bind(&Impl::processNextRequest, shared_from_this()));
            return;
        }
        auto session = getSession(request);
        doRequest(request, session);
    }

    void doRequest(RequestDataPtr request, SessionPtr session) {
        auto self = shared_from_this();
        if (session->isReusable()) {
            session->cancel([this, self, request](SessionPtr conn, error::Code errc) {
                if (!errc) {
                    conn->run(request, std::bind(&Impl::handleRun, self, p::_1, p::_2, p::_3));
                } else {
                    YLOG_CTX_LOCAL(request->ctx, info)
                        << "request wll be retried: conn=" << conn->id()
                        << ", errc='" << error::message(errc) << "'";
                    pendingRequests.push_front(std::move(request));
                    return processNextRequest();
                }
            });
        } else if (session->processedRequestsCount() == 0) {
            session->run(request, std::bind(&Impl::handleRun, self, p::_1, p::_2, p::_3));
        } else {    // session expired
            pendingRequests.push_front(std::move(request));
            processNextRequest();
        }
    }

    SessionPtr getSession(const RequestDataPtr& req) {
        if (!req->reuseConnection) {
            return std::make_shared<Session>(iodata, logger(), settings);
        }
        auto pool = getPool(req);
        return pool->getOrCreateSession([this]() {
            return std::make_shared<Session>(iodata, logger(), settings);
        });
    }

    void putSession(SessionPtr session, const RequestDataPtr& req) {
        auto pool = getPool(req);
        pool->put(session);
    }

    SessionPoolPtr getPool(const RequestDataPtr& req) {
        SessionInfo info{req->data.address, req->data.auth, req->useSsl};
        auto poolIt = connections.find(info);
        if (poolIt == connections.end()) {
            poolIt = connections.emplace(info, std::make_shared<SessionPool>(settings.maxPoolSize)).first;
        }
        return poolIt->second;
    }

    void handleRun(RequestDataPtr req, SessionPtr session, error::Code errc) {
        if (session->isReusable()) {
            auto self = shared_from_this();
            session->asyncWaitEOF([self](boost::system::error_code) { /*noop*/ });
            putSession(session, req);
        } else if (session->isOpen()) {
            auto self = shared_from_this();
            session->asyncShutdown([self](error::Code) { /*noop*/ });
        }
        get_io_service()->post([req, errc]() { req->callback(errc, std::move(req->response)); });
        processNextRequest();
    }

private:
    yplatform::net::io_data iodata;
    Settings settings;
    Connections connections;
    std::deque<RequestDataPtr> pendingRequests;
};


Client::Client(
    boost::asio::io_service& ios,
    const yplatform::log::source& logger,
    const Settings& settings
)
    : yplatform::log::contains_logger(logger)
    , impl(std::make_shared<class Impl>(ios, logger, settings))
{}

void Client::asyncRun(Context ctx, Request request, Callback callback) {
    asyncRun(ctx, std::move(request), Options(), std::move(callback));
}

void Client::asyncRun(Context ctx, Request request, Options opts, Callback callback) {
    impl->get_io_service()->post(std::bind(&Impl::asyncRun,
        impl, ctx, std::move(request), std::move(opts), std::move(callback))
    );
}

SmtpSessionPtr Client::createSmtpSession(Context ctx) {
    return impl->createSmtpSession(ctx);
}

SmtpSessionPtr Client::createSmtpSession(Context ctx, const Timeouts& timeouts) {
    return impl->createSmtpSession(ctx, timeouts);
}

}   // namespace ymod_smtpclient
