#include "command_composer.h"

#include <boost/asio/ip/host_name.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/algorithm/string/trim.hpp>

namespace ymod_smtpclient {

Command Composer::HELLO(SmtpPoint::Proto proto, const std::string& hostname) {
    Command result;
    result.name = ((proto == SmtpPoint::lmtp) ? "LHLO" : "EHLO");
    result.args = !hostname.empty() ? hostname : boost::asio::ip::host_name();
    return result;
}

Command Composer::AUTH(sasl::Mechanism mechanism) {
    Command command;
    command.name = "AUTH " + to_string(mechanism);
    return command;
}

std::string remove_brackets(const std::string& email) {
    auto filtered = boost::trim_copy(email);
    if (!filtered.empty() && filtered.front() == '<' && filtered.back() == '>') {
        return filtered.substr(1, filtered.length() - 2);
    }
    return filtered;
}

Command Composer::MAIL_FROM(const MailFrom& mailfrom) {
    Command result;
    result.name = "MAIL FROM";
    result.args = "<" + remove_brackets(mailfrom.email) + ">";
    if (mailfrom.envid) {
        result.args += " ENVID=" + *mailfrom.envid;
    }
    result.delim = ':';
    return result;
}

Command Composer::RCPT_TO(const RcptTo& rcpt, bool enableDsn) {
    Command result;
    result.name = "RCPT TO";
    result.args = "<" + rcpt.email + ">";
    if (enableDsn && !rcpt.notifyModes.empty()) {
        result.args += " NOTIFY=" + boost::join(rcpt.notifyModes | boost::adaptors::transformed(
            [](NotifyMode mode){ return to_string(mode); }), ",");
    }
    result.delim = ':';
    return result;
}

Command Composer::DATA_START() {
    Command command;
    command.name = "DATA";
    return command;
}

Command Composer::STARTTLS() {
    Command command;
    command.name = "STARTTLS";
    return command;
}

Command Composer::RSET() {
    Command command;
    command.name = "RSET";
    return command;
}

Command Composer::QUIT() {
    Command command;
    command.name = "QUIT";
    return command;
}

Command Composer::WITHOUT_NAME(const std::string& args, boost::optional<std::string> debug) {
    Command result;
    result.args = args;
    result.debugArgs = debug;
    return result;
}

}   // namespace ymod_smtpclient
