#include "response_reader.h"

namespace ymod_smtpclient {

ResponseReader::ResponseReader(
    Socket& socket,
    Buffer& readBuf,
    Duration timeout,
    const ContextPtr& ctx,
    const yplatform::log::source& logger,
    CompletionHandler&& handler
)
    : yplatform::log::contains_logger(logger)
    , socket(socket)
    , readBuf(readBuf)
    , timeout(timeout)
    , completionHandler(handler)
    , ctx(ctx) {}

void ResponseReader::operator() (YieldCtx yieldCtx, error::Code errc, std::string line) {
    bool isLastLine = false;
    reenter(yieldCtx) {
        for (;;) {
            yield asyncReadLine(timeout, yieldCtx);
            if (errc) {
                yield break;
            }
            try {
                isLastLine = updateMultilineResponse(line);
            } catch (const std::exception& e) {
                YLOG_CTX_LOCAL(ctx, error)
                    << "Unable to parse server response"
                    << ", reason: " << e.what();
                errc = error::ParseResponseError;
            }
            if (!isLastLine) {
                continue;
            }
            yield break;
        }
    }
    if (yieldCtx.is_complete()) {
        completionHandler(errc, resp);
    }
}

void ResponseReader::asyncReadLine(Duration timeout, YieldCtx handler) {
    socket.async_read_until(readBuf, smtpFilter, timeout,
        [this, handler](auto errc, auto /*size*/) mutable {
            if (errc) {
                auto status = (errc == boost::asio::error::operation_aborted)
                    ? error::RequestTimedOut
                    : error::ReadError;
                return handler(status, std::string());
            }
            std::string line;
            try {
                std::istream stream(&readBuf);
                std::getline(stream, line);
            } catch (const std::exception& e) {
                YLOG_CTX_LOCAL(ctx, error)
                    << "Unable to read line from server response"
                    << ", reason: " << e.what();
                return handler(error::ParseResponseError, std::string());
            }
            handler(error::Success, line);
        });
}

bool ResponseReader::updateMultilineResponse(const std::string& line) {
    server::OneLineResponse oneLine;
    bool isLast = false;
    std::tie(oneLine, isLast) = server::parseResponse(line);
    resp.dataLines.emplace_back(oneLine.data);
    if (isLast) {
        resp.replyCode = oneLine.replyCode;
        resp.enhancedCode = oneLine.enhancedCode;
    }
    return isLast;
}

}  // namespace ymod_smtpclient
