#include "client.h"
#include "../command_composer.h"

#include <yplatform/encoding/base64.h>

#include <stdexcept>

namespace ymod_smtpclient {

std::string to_string(sasl::Mechanism mechanism) {
    switch(mechanism) {
    case sasl::Mechanism::Login:
        return "LOGIN";
    case sasl::Mechanism::Plain:
        return "PLAIN";
    case sasl::Mechanism::Xoauth2:
        return "XOAUTH2";
    case sasl::Mechanism::None:
        return "None";
    default:
        throw std::runtime_error("invalid auth mechanism value: " + std::to_string(int(mechanism)));
    }
}

namespace sasl {

enum class ClientEngine::State : std::uint8_t {
    Start = 0,
    AfterCommand,
    AfterLogin,
    AfterPassword,
    Done,
    Error
};

ClientEngine::ClientEngine(std::size_t maxCommandLength)
    : maxCommandLength(maxCommandLength)
    , state(State::Start)
{}

void ClientEngine::reset(const AuthData& data) {
    authdata = data;
    state = State::Start;
}

ClientEngine::Status ClientEngine::next(
    const MultiLineResponse& response, Command& command)
{
    command = Command();
    switch (authdata.mechanism) {
    case Mechanism::Login:
        return handleLogin(response, command);
    case Mechanism::Plain:
        return handlePlain(response, command);
    case Mechanism::Xoauth2:
        return handleOauth(response, command);
    default:
        return Status::Done;
    }
}

ClientEngine::Status ClientEngine::handleLogin(
    const MultiLineResponse& response, Command& command)
{
    switch (state) {
    case State::Start:
        {
            command = Composer::AUTH(authdata.mechanism);
            state = State::AfterCommand;
            return Status::More;
        }
    case State::AfterCommand:
        {
            if (response.replyCode == 334) {
                // Check if this is the last response line.
                auto enc = yplatform::base64_encode(authdata.login.begin(), authdata.login.end());
                command = Composer::WITHOUT_NAME(std::string(enc.begin(), enc.end()), authdata.login);
                state = State::AfterLogin;
                return Status::More;
            }
            state = State::Error;
            return Status::Form;
        }
    case State::AfterLogin:
        {
            if (response.replyCode == 334) {
                auto enc = yplatform::base64_encode(authdata.password.begin(), authdata.password.end());
                command = Composer::WITHOUT_NAME(
                    std::string(enc.begin(), enc.end()), std::string("<password>"));
                state = State::AfterPassword;
                return Status::More;
            }
            state = State::Error;
            return Status::Form;
        }
    case State::AfterPassword:
        {
            // Check if they want something else.
            if (response.replyCode == 334) {
                state = State::Error;
                return Status::Form;
            }
            state = State::Done;
            return Status::Done;
        }
    case State::Done:
        return Status::Done;
    default:
        return Status::Form;
    }
}

ClientEngine::Status ClientEngine::handlePlain(
    const MultiLineResponse& response, Command& command)
{
    switch (state) {
    case State::Start:
        {
            command = Composer::AUTH(authdata.mechanism);
            command.args = makePlain(authdata.login, authdata.password);
            command.debugArgs = "<secret>";

            state = State::AfterPassword;
            if (command.str().size() > maxCommandLength) {
                command = Composer::AUTH(authdata.mechanism);
                state = State::AfterCommand;
            }
            return Status::More;
        }
    case State::AfterCommand:
        {
            if (response.replyCode == 334) {
                command = Composer::WITHOUT_NAME(
                    makePlain(authdata.login, authdata.password), std::string("<secret>"));
                state = State::AfterPassword;
                return Status::More;
            }
            state = State::Error;
            return Status::Form;
        }
    case State::AfterPassword:
        {
            // Check if they want something else.
            if (response.replyCode == 334) {
                state = State::Error;
                return Status::Form;
            }
            state = State::Done;
            return Status::Done;
        }
    case State::Done:
        return Status::Done;
    default:
        return Status::Form;
    }
}

ClientEngine::Status ClientEngine::handleOauth(
    const MultiLineResponse& response, Command& command)
{
    switch (state) {
    case State::Start:
        {
            command = Composer::AUTH(authdata.mechanism);
            command.args = makeOauthBearer(authdata.login, authdata.token);
            command.debugArgs = "<secret>";

            state = State::AfterPassword;
            if (command.str().size() > maxCommandLength) {
                command = Composer::AUTH(authdata.mechanism);
                state = State::AfterCommand;
            }
            return Status::More;
        }
    case State::AfterCommand:
        {
            if (response.replyCode == 334) {
                command = Composer::WITHOUT_NAME(
                    makeOauthBearer(authdata.login, authdata.token), std::string("<secret>"));
                state = State::AfterPassword;
                return Status::More;
            }
            state = State::Error;
            return Status::Form;
        }
    case State::AfterPassword:
        {
            // Check if they want something else.
            if (response.replyCode == 334) {
                state = State::Error;
                return Status::Form;
            }
            state = State::Done;
            return Status::Done;
        }
    case State::Done:
        return Status::Done;
    default:
        return Status::Form;
    }
}

inline std::string ClientEngine::makePlain(const std::string& login, const std::string& password) {
    auto line = login + '\0' + login + '\0' + password;
    auto encoded = yplatform::base64_encode(line.begin(), line.end());
    return std::string(encoded.begin(), encoded.end());
}

inline std::string ClientEngine::makeOauthBearer(const std::string& user, const std::string& token) {
    auto line = "user=" + user + "\1auth=Bearer " + token + "\1\1";
    auto encoded = yplatform::base64_encode(line.begin(), line.end());
    return std::string(encoded.begin(), encoded.end());
}

}   // namespace sasl
}   // namespace ymod_smtpclient
