#include "server_info.h"

#include <ymod_smtpclient/auth.h>

#include <boost/algorithm/string.hpp>
#include <boost/spirit/include/qi.hpp>
#include <boost/spirit/include/qi_omit.hpp>

#include <stdexcept>
#include <set>

namespace ymod_smtpclient {

void parse_auth_mechanisms(const std::string& line, std::set<sasl::Mechanism>& mechanisms) {
    std::vector<std::string> tokens;
    boost::split(tokens, line, boost::is_any_of(" "), boost::token_compress_on);

    for (auto it = tokens.begin() + 1; it != tokens.end(); ++it) {
        if (boost::iequals(*it, "LOGIN")) {
            mechanisms.insert(sasl::Mechanism::Login);
        } else if (boost::iequals(*it, "PLAIN")) {
            mechanisms.insert(sasl::Mechanism::Plain);
        } else if (boost::iequals(*it, "XOAUTH2")) {
            mechanisms.insert(sasl::Mechanism::Xoauth2);
        }
    }
}

namespace qi = boost::spirit::qi;

boost::optional<std::size_t> parse_size(const std::string& line) {
    boost::optional<std::size_t> result;
    auto beg = line.begin();
    auto end = line.end();
    if (qi::phrase_parse(beg, end, qi::omit[qi::no_case["SIZE"]] >> -qi::uint_, qi::space, result) &&
        (beg == end))
    {
        return result;
    }
    throw std::runtime_error("incorrect extension line: '" + line + "'");
}

ServerExtensions parse_smtp_server_extensions(const MultiLineResponse& response) {
    ServerExtensions result;
    for (const auto& line: response.dataLines) {
        auto stripped = boost::to_upper_copy(boost::trim_copy(line));
        if (boost::starts_with(stripped, "PIPELINING")) {
            result.pipelining = true;
        } else if (boost::starts_with(stripped, "DSN")) {
            result.dsn = true;
        } else if (boost::starts_with(stripped, "CHUNKING")) {
            result.chunking = true;
        } else if (boost::starts_with(stripped, "STARTTLS")) {
            result.starttls = true;
        } else if (boost::starts_with(stripped, "SMTPUTF8")) {
            result.utf8Enable = true;
        } else if (boost::starts_with(stripped, "ENHANCEDSTATUSCODES")) {
            result.enhancedStatusCodes = true;
        } else if (boost::starts_with(stripped, "SIZE")) {
            result.maxSize = parse_size(stripped);
        } else if (boost::starts_with(stripped, "AUTH")) {
            parse_auth_mechanisms(stripped, result.authMechanisms);
        }
    }
    return result;
}

}   // namespace ymod_smtpclient
