#include "server_response.h"

#include <boost/spirit/include/qi.hpp>
#include <boost/spirit/include/qi_optional.hpp>
#include <boost/spirit/include/qi_omit.hpp>
#include <boost/fusion/adapted/struct.hpp>
#include <boost/spirit/include/phoenix.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/algorithm/string/trim.hpp>

BOOST_FUSION_ADAPT_STRUCT(
    ymod_smtpclient::server::OneLineResponse, replyCode, enhancedCode, data)

namespace ymod_smtpclient {
namespace server {

namespace qi = boost::spirit::qi;

template <typename Iterator>
struct ResponseParser : qi::grammar<Iterator, OneLineResponse()> {
    ResponseParser(): ResponseParser::base_type(start) {
        using boost::phoenix::construct;
        using boost::phoenix::static_cast_;
        using qi::uint_parser;
        using qi::omit;
        using qi::blank;
        using qi::char_;

        uint_parser<std::uint16_t, 10, 1, 3> uint1_3;
        uint_parser<std::uint16_t, 10, 3, 3> uint3_3;

        replyCode = uint3_3;

        enhanced = (qi::char_("245") >> "." >> uint1_3 >> "." >> uint1_3)[
             qi::_val = construct<EnhancedStatusCode>(
                 static_cast_<EnhancedStatusCode::ClassType>(qi::_1 - '0'),
                 static_cast_<EnhancedStatusCode::SubjectType>(qi::_2),
                 qi::_3)];

        data = *(~qi::ascii::char_("\r\n"));

        start = replyCode
            >> (omit["-"] | (-(omit[+blank] >> enhanced) >> omit[+blank]))
            >> -data >> omit[*qi::eol];
    }

private:
    qi::rule<Iterator, std::uint16_t()> replyCode;
    qi::rule<Iterator, EnhancedStatusCode()> enhanced;
    qi::rule<Iterator, std::string()> data;
    qi::rule<Iterator, OneLineResponse()> start;
};

std::pair<OneLineResponse, bool> parseResponse(const std::string& response) {
    bool isTheLastLine = true;
    if (response.size() >= 4 && response[3] == '-') {
        isTheLastLine = false;
    }
    OneLineResponse result;
    ResponseParser<std::string::const_iterator> parser;

    auto beg = response.begin();
    auto end = response.end();

    if (!qi::parse(beg, end, parser, result) || (beg != end)) {
        throw std::runtime_error("invalid format: '" + boost::trim_copy(response) + "'");
    }
    if (result.enhancedCode &&
        result.enhancedCode.get().classType == EnhancedStatusCode::ClassType::Unknown)
    {
        result.enhancedCode.reset();
    }

    return std::make_pair(std::move(result), isTheLastLine);
}

}   // namespace server

OneResponse multiline2OneLineResponse(const MultiLineResponse& response) {
    OneResponse result;
    result.replyCode = response.replyCode;
    result.enhancedStatusCode = response.enhancedCode;
    result.data = boost::join(response.dataLines, "; ");
    return result;
}

}   // namespace ymod_smtpclient


