#pragma once

#include "smtp_session_impl.h"

#include <yplatform/time_traits.h>

#include <boost/algorithm/string/trim.hpp>

namespace ymod_smtpclient {

class Session
    : public std::enable_shared_from_this<Session>
    , public yplatform::log::contains_logger
    , public boost::noncopyable
{
public:
    using ThisPtr = std::shared_ptr<Session>;
    using CompletionHandler = std::function<void (RequestDataPtr, ThisPtr, error::Code)>;
    using CancelHandler = std::function<void (ThisPtr, error::Code)>;
    using Response = MultiLineResponse;

    Session(yplatform::net::io_data& iodata,
        const yplatform::log::source& logger,
        const Settings& settings);

    ~Session();

    template <typename Handler>
    void run(RequestDataPtr req, Handler&& handler);

    template <typename Handler>
    void cancel(Handler&& handler);

    template <typename Handler>
    void asyncWaitEOF(Handler&& handler);

    template <typename Handler>
    void asyncShutdown(Handler&& handler);

    void close();

    bool isOpen() const { return session->isOpen(); }
    bool isReusable() const { return reusable && isOpen(); }
    bool isAuthorized() const { return authorized; }

    void const* id() const { return session->getId(); }

    std::uint32_t processedRequestsCount() const { return requestsProcessed; }

private:
    void runCoro();

    void connect();

    void greeting();
    void hello();
    void starttls();
    void rset();
    void pipelineCommands();

    void authorize();

    void handleMailfrom(error::Code, const Response& resp);
    void handleRcptTo(error::Code, const Response& resp);
    void handleDataStart(error::Code, const Response& resp);

    void handleSmtpResponse(error::Code errc, const Response& resp);
    void handleLmtpResponse(error::Code errc, const Response& resp);

    error::Code calcSessionErrorCode();

    void complete(error::Code errc, const std::string& reason = std::string());
    void complete(error::Code errc, const Response& resp, const std::string& reason);

    bool checkCorrectEOF(boost::system::error_code errc);

private:
    const Settings& settings;

    bool authorized = false;
    bool reusable = false;
    yplatform::time_traits::time_point createdAt = yplatform::time_traits::clock::now();

    error::Code lastErrorCode = error::Code::Success;
    std::uint32_t requestsProcessed = 0;

    boost::asio::coroutine coro;

    RequestDataPtr request;
    CompletionHandler completionHandler;
    std::vector<RcptTo>::iterator rcptActive;

    CancelHandler cancelHandler;

    yplatform::log::source clientLog;
    yplatform::log::source serverLog;

    std::shared_ptr<SmtpSessionImpl> session;
};

using SessionPtr = std::shared_ptr<Session>;
using SessionWeakPtr = std::weak_ptr<Session>;

template <typename Handler>
void Session::run(RequestDataPtr req, Handler&& handler) {
    request = req;
    completionHandler = std::forward<Handler>(handler);
    request->startedAt = yplatform::time_traits::clock::now();
    coro = boost::asio::coroutine{};
    clientLog.set_log_prefix(logger().get_log_prefix() + request->ctx->uniq_id() + " client:");
    serverLog.set_log_prefix(logger().get_log_prefix() + request->ctx->uniq_id() + " server:");
    reusable = request->reuseConnection;
    ++requestsProcessed;
    session->setTimeouts(request->timeouts);
    session->setContext(request->ctx);
    runCoro();
}
template <typename Handler>
void Session::cancel(Handler&& handler) {
    session->cancel();
    cancelHandler = std::forward<Handler>(handler);
}

template <typename Handler>
void Session::asyncWaitEOF(Handler&& waitCb) {
    auto self = shared_from_this();
    session->asyncWaitEof(settings.sessionMaxIdleTime,
        [self, this, waitCb = std::move(waitCb)] (boost::system::error_code errc) {
            CancelHandler cancelCb(std::move(cancelHandler));
            if ((errc == boost::asio::error::operation_aborted) && cancelCb) {
                cancelCb(self, error::Code::Success);
            } else {
                close();
                if (cancelCb) {
                    cancelCb(self, error::Code::ConnectionExpired);
                }
            }
            waitCb(errc);
    });
}

template <typename Handler>
void Session::asyncShutdown(Handler&& handler)  {
    auto self = shared_from_this();
    session->shutdown();
    session->asyncWaitEof(settings.sessionMaxIdleTime,
        [self, this, handler = std::move(handler)](boost::system::error_code errc) {
            handler(checkCorrectEOF(errc) ? error::Code::Success : error::Code::ConnectionClosedError);
    });
}

}   // namespace ymod_smtpclient
