#pragma once

#include "session.h"

#include <boost/noncopyable.hpp>
#include <algorithm>
#include <deque>

namespace ymod_smtpclient {

class SessionPool
    : public std::enable_shared_from_this<SessionPool>
    , private boost::noncopyable
{
public:
    using duration = yplatform::time_traits::duration;
    using time_point = yplatform::time_traits::time_point;

    SessionPool(uint16_t maxPoolSize = 10): maxPoolSize(maxPoolSize) {}

    template <typename Creator>
    SessionPtr getOrCreateSession(Creator creator) {
        while (!sessions.empty()) {
            auto sess = sessions.front();
            sessions.pop_front();
            if (!sess.expired() && sess.lock()->isReusable()) {
                return sess.lock();
            }
        }
        return creator();
    }

    void put(SessionPtr session) {
        if (!session || !session->isReusable()) {
            return;
        }
        if (sessions.size() >= maxPoolSize) {
            if (!sessions.back().expired()) {
                sessions.back().lock()->close();
            }
            sessions.pop_back();
        }
        sessions.emplace_front(session);
    }

    uint16_t size() const { return static_cast<uint16_t>(sessions.size()); }

private:
    using Sessions = std::deque<SessionWeakPtr>;

    uint16_t maxPoolSize;
    Sessions sessions;
};


}   // namespace ymod_smtpclient
