#include <ymod_smtpclient/smtp_point.h>

#include <boost/spirit/include/qi.hpp>
#include <boost/spirit/include/qi_optional.hpp>
#include <boost/spirit/include/qi_omit.hpp>

#include <boost/fusion/adapted/struct.hpp>

BOOST_FUSION_ADAPT_STRUCT(
    ymod_smtpclient::SmtpPoint,
    (ymod_smtpclient::SmtpPoint::Proto, proto),
    (std::string, host),
    (unsigned short, port))

namespace ymod_smtpclient {

namespace qi = boost::spirit::qi;

template <typename Iterator>
struct SmtpPointParser: qi::grammar<Iterator, SmtpPoint()> {
    SmtpPointParser(): SmtpPointParser::base_type(address) {
        using qi::char_;
        using qi::int_;
        using qi::lit;
        using qi::omit;
        using qi::no_case;

        proto.add("smtp", SmtpPoint::Proto::smtp)("lmtp", SmtpPoint::Proto::lmtp);

        bracket_host = (omit[char_('[')] >> +(~char_("[]")) >> omit[char_(']')]);
        single_host = +(~char_(":/[]"));
        host = bracket_host | single_host;

        address = -(no_case[proto] >> omit[lit("://")]) >> host >> -(omit[char_(':')] >> int_);
    }

private:
    qi::symbols<char, SmtpPoint::Proto> proto;
    qi::rule<Iterator, std::string()> host, bracket_host, single_host;
    qi::rule<Iterator, SmtpPoint()> address;
};


SmtpPoint SmtpPoint::fromString(const std::string& info) {
    SmtpPointParser<std::string::const_iterator> parser;
    SmtpPoint result;
    auto first = info.begin();
    auto last = info.end();
    if (!qi::parse(first, last, parser, result) || (first != last)) {
        throw std::logic_error("invalid syntax of 'smtp point': " + info);
    }
    return result;
}

}   // namespace ymod_smtpclient

