#pragma once

#include "types.h"
#include "request_data.h"
#include "server_response.h"
#include "server_info.h"
#include "sasl/client.h"
#include "command_composer.h"
#include "dot_stuffer.h"
#include "response_reader.h"

#include <ymod_smtpclient/smtp_session.h>

#include <yplatform/net/socket.h>
#include <yplatform/net/sequental_connect.h>
#include <yplatform/net/settings.h>
#include <yplatform/net/io_data.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace ymod_smtpclient {

class SmtpSessionImpl
    : public std::enable_shared_from_this<SmtpSessionImpl>
    , public boost::noncopyable
    , public yplatform::log::contains_logger
    , public SmtpSession
{
public:

    SmtpSessionImpl(
        yplatform::net::io_data& ioData,
        const Settings& settings,
        const yplatform::log::source& logger);

    SmtpSessionImpl(
        yplatform::net::io_data& ioData,
        const Settings& settings);

    ~SmtpSessionImpl() override;

    void setContext(const ContextPtr& newCtx);

    void setTimeouts(const Timeouts& newTimeouts);

    void cancel() override;

    void close() override;

    void shutdown(bool gracefull=true) override;

    bool isOpen() const override;

    bool isEncrypted() const override;

    boost::asio::ip::address getLocalAddr() const;

    boost::asio::ip::address getRemoteAddr() const;

    uint16_t getLocalPort() const;

    uint16_t getRemotePort() const;

    void const* getId() const;

    ServerExtensions getServerExtensions() const override;

    void asyncConnect(std::string host, uint16_t port, Optional<bool> useSsl, Handler handler) override;

    void asyncGreeting(SmtpHandler handler) override;

    void asyncHelo(SmtpPoint::Proto proto, const std::string& hostname, SmtpHandler handler) override;

    void asyncHelo(SmtpPoint::Proto proto, SmtpHandler handler) override;

    void asyncAuth(const AuthData& authData, SmtpHandler handler) override;

    void asyncStartTls(SmtpHandler handler) override;

    void asyncMailFrom(const MailFrom& mailFrom, SmtpHandler handler) override;

    void asyncRcptTo(const RcptTo& rcptTo, bool enableDsn, SmtpHandler handler) override;

    void asyncDataStart(SmtpHandler handler) override;

    void asyncRset(SmtpHandler handler) override;

    void asyncQuit(SmtpHandler handler) override;

    void asyncWriteMessage(const std::string& msg, bool enableDotStuffing, SmtpHandler handler) override;

    void asyncWriteCommand(const std::string& cmd, Handler handler);

    void asyncReadMessageResponse(SmtpHandler handler);

    void asyncHandleMailFrom(SmtpHandler handler);

    void asyncHandleRcptTo(SmtpHandler handler);

    void asyncHandleDataStart(SmtpHandler handler);

    template <class T>
    void asyncWaitEof(Duration timeout, T handler);

private:
    void asyncTlsHandshake(Handler handler);

    void asyncWrite(const std::string& buffer, TimePoint deadline, Handler handler);

    void asyncExecCommand(const Command& cmd, SmtpHandler handler);

    void asyncExecCommand(const Command& cmd, uint16_t expectedReplyCode, SmtpHandler handler);

    void asyncReadResponse(Duration timeout, SmtpHandler handler);

    void asyncReadCommandResponse(SmtpHandler handler);

    void asyncHandleCommandResponse(uint16_t expectedReplyCode, SmtpHandler handler);

    void asyncAuthProceed(Response authResp, SmtpHandler handler);

    bool updateServerExtensions(const Response& resp);

    void clearBuffers();

    bool isSslRequired(uint16_t port, Optional<bool> useSsl) const;

    Socket socket;
    Buffer readBuf;
    Buffer writeBuf;

    yplatform::net::async_sequental_connect_op<Socket> connectOp;

    ServerExtensions serverExtensions;
    sasl::ClientEngine authEngine;

    Timeouts timeouts;

    ContextPtr ctx;
    uint16_t smtpSslDefaultPort;
    bool encrypted = false;
};

template <class T>
void SmtpSessionImpl::asyncWaitEof(Duration timeout, T handler)  {
    auto self = shared_from_this();
    socket.async_read(readBuf, timeout,
        [self, handler](boost::system::error_code errc, std::size_t) {
            handler(errc);
        }, /*transfer_at_least=*/1U);
}

}  // namespace ymod_smtpclient
