#pragma once

#include <yplatform/util/split.h>
#include <yplatform/log.h>
#include <boost/optional.hpp>
#include <string>

namespace ymod_smtpserver {

struct EnhancedStatusCode {
    enum class ClassType {
        Success = 2,
        PersistentTransientFailure = 4,
        PermanentFailure = 5
    };

    enum class SubjectType {
        Other                  = 0,
        Addressing             = 1,
        Mailbox                = 2,
        MailSystem             = 3,
        NetworkAndRouting      = 4,
        MailDeliveryProtocol   = 5,
        MessageContent         = 6,
        Security               = 7
    };

    EnhancedStatusCode(ClassType c, SubjectType s, int detail)
        : classType(c)
        , subjectType(s)
        , detail(detail)
    {}

    EnhancedStatusCode(int code)
        : classType(static_cast<ClassType>(code / 100))
        , subjectType(static_cast<SubjectType>((code / 10) % 10))
        , detail(code % 10)
    {}

    ClassType classType;
    SubjectType subjectType;
    int detail;
};

template <typename Enumeration>
inline typename std::underlying_type<Enumeration>::type as_integer(Enumeration const value) {
    return static_cast<typename std::underlying_type<Enumeration>::type>(value);
}

inline std::ostream& operator<< (std::ostream& os, const EnhancedStatusCode& code) {
    os << as_integer(code.classType) << "." << as_integer(code.subjectType) << "." << code.detail;
    return os;
}

inline std::string response_description(int code) {
    switch (code) {
    case 221:
        return "Closing connecton";
    case 250:
        return "Ok";
    case 451:
        return "Sorry, service unavailable";
    case 502:
        return "Unrecognized command";
    case 503:
        return "Bad sequence of commands";
    case 555:
        return "Syntax error";
    default:
        return "Unknown error";
    }
}

struct Response {
    using EnhancedStatusCodeOpt = boost::optional<EnhancedStatusCode>;

    int code = 0;
    EnhancedStatusCodeOpt enhancedCode;
    std::string text;

    Response() = default;

    Response(int code, const std::string& text, EnhancedStatusCodeOpt edCode = EnhancedStatusCodeOpt())
        : code(code)
        , enhancedCode(edCode)
        , text(text)
    {}

    Response(int code, EnhancedStatusCodeOpt edCode = EnhancedStatusCodeOpt())
        : code(code)
        , enhancedCode(edCode)
        , text(response_description(code))
    {}

    Response operator()(const std::string& desc) const { return Response(code, desc, enhancedCode); }
    Response operator()(EnhancedStatusCode edCode) const { return Response(code, text, edCode); }

    bool operator== (const Response& rh) const { return code == rh.code; }
};

inline std::ostream& operator<< (std::ostream& os, const Response& resp) {
    if (resp.text.empty()) {
        os << resp.code;
        if (resp.enhancedCode) {
            os << ' ' << *resp.enhancedCode;
        }
        os << "\r\n";
        return os;
    }
    auto lines = yplatform::util::split(resp.text, "\n");
    for (std::size_t i = 0; i < lines.size(); ++i) {
        os << resp.code << (((i + 1) == lines.size() ? ' ' : '-'));
        if (resp.enhancedCode) {
            os << resp.enhancedCode.get() << ' ';
        }
        os << lines[i] << "\r\n";
    }
    return os;
}

}   // namespace ymod_smtpserver
