#include "connection_impl.h"
#include "parser/command.h"
#include "parser/line.h"
#include <mail/ymod_smtpserver/src/utils.h>

#include <boost/spirit/include/qi.hpp>
#include <boost/algorithm/string.hpp>

namespace ymod_smtpserver {

namespace qi = boost::spirit::qi;

void ConnectionImpl::close() {
    auto self(shared_from_this());
    ios.post([this, self]() {
        socket.cancel_operations();
        socket.shutdown(true);
        socket.close();
    });
}

void ConnectionImpl::tlsHandshake(TlsHandshakeCallback callback) {
    auto self(shared_from_this());
    ios.post([this, self, callback = std::move(callback)]() {
        secure = true;
        socket.async_tls_handshake(
            yplatform::net::tcp_socket::handshake_type::server,
            settings->socket.tls_timeout,
            std::move(callback));
    });
}

SSL* ConnectionImpl::getSsl() {
    if (!isSecure()) {
        return nullptr;
    }
    return socket.stream().get_ssl_stream()->native_handle();
}

void ConnectionImpl::writeResponse(Response response, WriteResponseCallback callback) {
    bufferResponse(std::move(response));
    auto self(shared_from_this());
    ios.post([this, self, callback = std::move(callback)]() {
        socket.async_write(writeBuf, settings->timeouts.writeResponse, std::move(callback));
    });
}

void ConnectionImpl::bufferResponse(Response response) {
    std::ostream os(&writeBuf);
    os << response;
}

void ConnectionImpl::writeResponses(const std::vector<Response>& responses, WriteResponseCallback callback) {
    bufferResponses(responses);
    auto self(shared_from_this());
    ios.post([this, self, callback = std::move(callback)]() {
        socket.async_write(writeBuf, settings->timeouts.writeResponse, std::move(callback));
    });
}

void ConnectionImpl::bufferResponses(const std::vector<Response>& responses) {
    std::ostream os(&writeBuf);
    for (const auto& response: responses) {
        os << response;
    }
}

struct ConnectionImpl::CommandHandler {
    explicit CommandHandler(ConnectionImplPtr conn, Deadline deadline, ReadCommandCallback callback)
        : conn(conn)
        , deadline(deadline)
        , callback(std::move(callback))
    {}

    void operator()(boost::system::error_code ec) {
        if (ec) {
            return callback(ec, commands::Unknown());
        }
        auto beg = boost::asio::buffers_begin(conn->readBuf.data());
        auto end = boost::asio::buffers_end(conn->readBuf.data());
        auto lineEnd = beg;
        if (parser::parse_line(lineEnd, end)) {
            auto lineSize = std::distance(beg, lineEnd);
            parse_command(beg, lineEnd, [this, lineSize, ec](Command cmd) {
                conn->readBuf.consume(lineSize);
                callback(ec, cmd);
            });
        } else {
            conn->readChunk(deadline, *this);   // need more data
        }
    }

private:
    ConnectionImplPtr conn;
    Deadline deadline;
    ReadCommandCallback callback;
};

void ConnectionImpl::readCommand(ReadCommandCallback callback) {
    auto self(shared_from_this());
    ios.post([this, self, callback = std::move(callback)](){
        auto deadline = yplatform::time_traits::clock::now() + settings->timeouts.readCommand;
        CommandHandler handler(self, deadline, std::move(callback));
        if (boost::asio::buffer_size(readBuf.data())) {
            handler(boost::system::error_code());   // parse non-empty buffer
        } else {
            readChunk(deadline, std::move(handler));
        }
    });
}

struct ConnectionImpl::MessageHandler {
    explicit MessageHandler(ConnectionImplPtr conn, Deadline deadline, ReadMessageCallback callback)
        : conn(conn)
        , deadline(deadline)
        , callback(std::move(callback))
    {}

    void operator()(boost::system::error_code ec) {
        if (ec) {
            return callback(ec, std::make_shared<std::string>());
        }
        auto beg = boost::asio::buffers_begin(conn->readBuf.data());
        auto end = boost::asio::buffers_end(conn->readBuf.data());
        auto processed = beg;
        auto finished = conn->msgParser.parse(beg, end, processed);
        conn->readBuf.consume(std::distance(beg, processed));
        if (finished) {
            auto message = conn->msgParser.getMessage();
            if (conn->settings->fixCrlf) {
                message = NUtil::FixCrlf(message->begin(), message->end());
            }
            conn->msgParser.reset();
            callback(ec, message);
        } else {
            conn->readChunk(deadline, *this);   // need more data
        }
    }

private:
    ConnectionImplPtr conn;
    Deadline deadline;
    ReadMessageCallback callback;
};

void ConnectionImpl::readMessage(ReadMessageCallback callback) {
    auto self(shared_from_this());
    ios.post([this, self, callback = std::move(callback)]() {
        auto deadline = yplatform::time_traits::clock::now() + settings->timeouts.readMessage;
        MessageHandler handler(self, deadline, std::move(callback));
        msgParser.reset();
        if (boost::asio::buffer_size(readBuf.data())) {
            handler(boost::system::error_code());   // parse non-empty buffer
        } else {
            readChunk(deadline, std::move(handler));
        }
    });
}

struct ConnectionImpl::ReadLineHandler {
    ReadLineHandler(ConnectionImplPtr conn, Deadline deadline, ReadLineCallback callback)
        : conn(conn)
        , deadline(deadline)
        , callback(std::move(callback))
    {}

    void operator()(boost::system::error_code ec) {
        if (ec) {
            return callback(ec, std::string{});
        }
        auto beg = boost::asio::buffers_begin(conn->readBuf.data());
        auto end = boost::asio::buffers_end(conn->readBuf.data());
        auto lineEnd = beg;
        if (parser::parse_line(lineEnd, end)) {
            auto lineSize = std::distance(beg, lineEnd);
            std::string line{beg, lineEnd};
            conn->readBuf.consume(lineSize);
            return callback(ec, std::move(line));
        } else {
            conn->readChunk(deadline, std::move(*this));
        }
    }

private:
    ConnectionImplPtr conn;
    Deadline deadline;
    ReadLineCallback callback;
};

void ConnectionImpl::readLine(ReadLineCallback callback) {
    ios.post([this, self = shared_from_this(), callback = std::move(callback)]() {
        auto deadline = yplatform::time_traits::clock::now() + settings->timeouts.readCommand;
        ReadLineHandler handler(self, deadline, callback);
        if (boost::asio::buffer_size(readBuf.data())) {
            handler(boost::system::error_code());   // parse non-empty buffer
        } else {
            readChunk(deadline, std::move(handler));
        }
    });
}

}   // namespace ymod_smtpserver
