#pragma once

#include <ymod_smtpserver/connection.h>
#include <ymod_smtpserver/commands.h>

#include "settings.h"
#include "parser/message.h"
#include "manager.h"

#include <yplatform/net/socket.h>
#include <yplatform/task_context.h>
#include <yplatform/context_repository.h>

#include <boost/noncopyable.hpp>
#include <memory>

namespace ymod_smtpserver {

class ConnectionImpl
    : public Connection
    , private boost::noncopyable
    , public std::enable_shared_from_this<ConnectionImpl>
{
public:
    ConnectionImpl(
        yplatform::net::tcp_socket&& socket,
        SettingsPtr settings,
        ConnectionManagerWeakPtr manager
    )
        : ios(*socket.get_io())
        , socket(std::move(socket))
        , settings(settings)
        , chunkSize(settings->chunkSize)
        , msgParser(settings->removeLeadingDots)
        , context(boost::make_shared<yplatform::task_context>())
        , startedAt(yplatform::time_traits::clock::now())
        , manager(manager)
    {
        yplatform::context_repository::instance().add_context(context);
    }

    ~ConnectionImpl() override {
        try {
            if (auto lock = manager.lock()) {
                lock->onDestroy(this);
            }
            yplatform::context_repository::instance().rem_context(context);
        } catch(...) {
        }
    }

    void close() override;

    bool isOpen() const override { return socket.is_open(); }
    bool isSecure() const override { return secure; }

    void tlsHandshake(TlsHandshakeCallback callback) override;
    SSL* getSsl() override;

    void readCommand(ReadCommandCallback callback) override;
    void readMessage(ReadMessageCallback callback) override;
    void readLine(ReadLineCallback callback) override;

    void writeResponse(Response response, WriteResponseCallback callback) override;
    void bufferResponse(Response response) override;

    void writeResponses(const std::vector<Response>& responses, WriteResponseCallback callback) override;
    void bufferResponses(const std::vector<Response>& responses) override;

    const boost::asio::ip::address& remoteAddr() const override { return socket.remote_addr(); }
    unsigned short remotePort() const override { return socket.remote_port(); }
    const boost::asio::ip::address& localAddr() const override { return socket.local_addr(); }
    unsigned short localPort() const override { return socket.local_port(); }
    const void* id() const override { return socket.id(); }

    boost::asio::io_service& ioService() override { return ios; }

    yplatform::task_context_ptr ctx() const override { return context; }
    yplatform::time_traits::time_point startTime() const override { return startedAt; }

private:
    using Deadline = yplatform::time_traits::time_point;

    struct CommandHandler;
    friend CommandHandler;

    struct MessageHandler;
    friend MessageHandler;

    struct ReadLineHandler;
    friend ReadLineHandler;

    template <typename Handler>
    void readChunk(Deadline deadline, Handler handler);

private:
    boost::asio::io_service& ios;
    yplatform::net::tcp_socket socket;
    SettingsPtr settings;
    OnCloseCallback onCloseCallback;

    boost::asio::streambuf readBuf;
    boost::asio::streambuf writeBuf;
    std::size_t chunkSize = amount::KB * 8;
    parser::MessageParser msgParser;

    bool secure = false;

    yplatform::task_context_ptr context;
    yplatform::time_traits::time_point startedAt;

    ConnectionManagerWeakPtr manager;
};

using ConnectionImplPtr = std::shared_ptr<ConnectionImpl>;

template <typename Handler>
void ConnectionImpl::readChunk(Deadline deadline, Handler handler) {
    socket.async_read(readBuf.prepare(chunkSize), deadline,
        [handler = std::move(handler), this](boost::system::error_code ec, std::size_t size) mutable {
            if (!ec) {
                readBuf.commit(size);
            } else if (isOpen()) {
                close();
            }
            handler(ec);
    });
}

}   // namespace ymod_smtpserver
