#include "impl.h"
#include "connection_impl.h"

namespace ymod_smtpserver {

void Impl::init(const yplatform::ptree& conf) {
    settings = std::make_shared<Settings>();
    settings->parsePtree(conf);

    reactor = yplatform::global_reactor_set->get(conf.get("reactor", "global"));
    for (size_t i = 0; i < reactor->size(); ++i) {
        if ((*reactor)[i]->size() != 1) {
            throw std::runtime_error("Multithreaded pool is not supported!");
        }
    }
    connManager = std::make_shared<ConnectionManager>(logger());
    boxes.reserve(reactor->size());
    for (std::size_t i = 0; i < reactor->size(); ++i) {
        boxes.emplace_back();
        auto& box = boxes.back();
        box.iodata = std::make_shared<yplatform::net::io_data>(*(*reactor)[i]->io());
        box.iodata->setup_ssl(settings->ssl);
        box.iodata->setup_dns();
        box.tcpServers.reserve(settings->endpoints.size());
    }
    for (const auto& ep: settings->endpoints) {
        YLOG_L(info) << "bind " << ep.addr << ":" << ep.port << " secure=" << ep.secure;
        for (auto& box: boxes) {
            box.tcpServers.emplace_back(*box.iodata, ep.addr, ep.port, settings->socket);
        }
    }
}

using TcpSocket = yplatform::net::tcp_socket;

void Impl::start() {
    for (std::size_t i = 0; i < settings->endpoints.size(); ++i) {
        const auto& ep = settings->endpoints[i];
        for (auto& box: boxes) {
            auto& server = box.tcpServers[i];
            auto self(shared_from_this());
            server.listen([this, self, ep](TcpSocket&& socket) {
                auto conn = std::make_shared<ConnectionImpl>(std::move(socket), settings, connManager);
                connManager->registerConnection(conn.get());
                if (ep.secure) {
                    conn->tlsHandshake([this, self, conn](boost::system::error_code ec) {
                        if (ec) {
                            YLOG_L(info) << "tls handshake error: " << ec.message();
                        } else {
                            YLOG_L(info) << "tls handshake ok";
                            auto session = factory->create(conn);
                            session->start();
                        }
                    });
                } else {
                    auto session = factory->create(conn);
                    session->start();
                }
            });
        }
    }
}

void Impl::stop() {
    for (auto& box: boxes) {
        for (auto& server: box.tcpServers) {
            server.stop();
        }
    }
}

}   // namespace ymod_smtpserver

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(ymod_smtpserver::Impl)
