#include "manager.h"

namespace ymod_smtpserver {

inline std::string get_remote_ip_string(boost::asio::ip::address ip) {
    if (ip.is_v6()) {
        boost::asio::ip::address_v6 v6 = ip.to_v6();
        if (v6.is_v4_mapped() || v6.is_v4_compatible()) {
            boost::asio::ip::address_v4 v4 = v6.to_v4();
            ip = v4;
        }
    }
    return ip.to_string();
}

using Clock = yplatform::time_traits::clock;

void ConnectionManager::registerConnection(Connection* conn) {
    YLOG_L(debug) << "conn=" << conn->id() << ", new connection from ["
        << get_remote_ip_string(conn->remoteAddr()) << "]:" << conn->remotePort();
    WriteLock lock(mutex);
    connections.insert(conn);
}

void ConnectionManager::onDestroy(Connection* conn) {
    YLOG_L(debug) << "conn=" << conn->id() << ", destroy connection"
        << ", duration=" << yplatform::time_traits::to_string(Clock::now() - conn->startTime());
    WriteLock lock(mutex);
    connections.erase(conn);
}

yplatform::ptree ConnectionManager::getStats() const {
    ReadLock lock(mutex);

    yplatform::ptree data;

    auto now = Clock::now();
    yplatform::ptree connectionsStat;
    for (const auto conn: connections) {
        yplatform::ptree one;
        one.put("conn", conn->id());
        one.put("port", conn->remotePort());
        one.put("host", get_remote_ip_string(conn->remoteAddr()));
        one.put("open", conn->isOpen());
        one.put("secure", conn->isSecure());
        one.put("duration", yplatform::time_traits::to_string(now - conn->startTime()));
        connectionsStat.push_back(std::make_pair("", one));
    }
    data.add_child("connections", connectionsStat);

    return data;
}

}   // namespace ymod_smtpserver
