#pragma once

#include <ymod_smtpserver/commands.h>

#include <boost/spirit/include/qi.hpp>
#include <boost/spirit/include/phoenix.hpp>
#include <boost/fusion/adapted/std_pair.hpp>

namespace ymod_smtpserver {
namespace parser {

namespace spirit = boost::spirit;
namespace qi = spirit::qi;

template <typename Iterator, typename Handler>
class CommandGrammar: public qi::grammar<Iterator> {
public:
    CommandGrammar(boost::iterator_range<Iterator> ctx, Handler handler)
        : qi::grammar<Iterator>(start)
        , handler(std::move(handler))
        , ctx(ctx)
    {
        using qi::ascii::char_;
        using qi::ascii::print;
        using qi::ascii::no_case;
        using qi::digit;
        using qi::alnum;
        using qi::cntrl;
        using qi::blank;
        using qi::attr;
        using qi::raw;
        using qi::repeat;
        using qi::int_;
        using qi::eps;
        using qi::_pass;;
        using qi::omit;
        using qi::eol;
        using boost::phoenix::construct;

        path = raw[+(qi::standard::char_ - qi::standard::blank - qi::standard::char_("<>") - cntrl)];
        ehlo_host = raw[+(qi::standard::print - qi::standard::blank)];
        esmtp_params = esmtp_pair % +blank;
        esmtp_pair = esmtp_key >> -('=' >> esmtp_value);
        esmtp_key = +(alnum >> *char_('-'));
        esmtp_value = +(print - cntrl - char_(" ="));

        mailfrom = (no_case["FROM:"] 
            >> omit[*blank] 
            >> (
                (omit[char_('<') >> *blank]
                >> (path | attr(std::string()))
                >> omit[*blank >> char_('>')])
                |
                (path | attr(std::string()))
            )
            >> (
                (omit[+blank] >> esmtp_params)
                |
                attr(commands::Params())
            )
            >> omit[*blank >> eol])
            [qi::_val = construct<commands::MailFrom>(qi::_1, qi::_2)];

        rcptto = (no_case["TO:"] 
            >> omit[*blank]
            >> (
                (omit[char_('<') >> *blank] 
                >> path
                >> omit[*blank >> char_('>')])
                |
                path
            )
            >> (
                (omit[+blank] >> esmtp_params)
                |
                attr(commands::Params())
            )
            >> omit[*blank >> eol])
            [qi::_val = construct<commands::RcptTo>(qi::_1, qi::_2)];

        ehlo = (ehlo_host >> omit[*blank >> eol])
            [qi::_val = construct<commands::Ehlo>(qi::_1)];
        lhlo = (ehlo_host >> omit[*blank >> eol])
            [qi::_val = construct<commands::Lhlo>(qi::_1)];
        helo = (ehlo_host >> omit[*blank >> eol])
            [qi::_val = construct<commands::Helo>(qi::_1)];

        data = (no_case["DATA"] >> omit[*blank >> eol]);
        quit = (no_case["QUIT"] >> omit[*blank >> eol]);
        rset = (no_case["RSET"] >> omit[*blank >> eol]);
        startTls = (no_case["STARTTLS"] >> omit[*blank >> eol]);
        noop = (no_case["NOOP"] >> omit[*(print - cntrl) >> eol]);

        auth = (omit[no_case["AUTH"] >> +blank] >> 
            qi::eps[boost::phoenix::ref(AuthMethod) = commands::AuthMethods::NotSupported{}] >>
            qi::no_case[
                qi::string("LOGIN")[boost::phoenix::ref(AuthMethod) = commands::AuthMethods::Login{}]
                | qi::string("PLAIN")[boost::phoenix::ref(AuthMethod) = commands::AuthMethods::Plain{}]
                | qi::string("XOAUTH2")[boost::phoenix::ref(AuthMethod) = commands::AuthMethods::XOAuth2{}]
                | omit[+(char_ - blank - cntrl)] >> qi::eps[boost::phoenix::ref(AuthMethod) = commands::AuthMethods::NotSupported{}]
            ] >>
            eps[boost::phoenix::ref(AuthInitialResponse) = std::string{}] >>
            -(omit[+blank] >> raw[+(char_ - blank - cntrl)][boost::phoenix::ref(AuthInitialResponse) = construct<std::string>(boost::phoenix::begin(qi::_1), boost::phoenix::end(qi::_1))]) >>
            omit[*blank >> eol]
        )[qi::_val = construct<commands::Auth>(boost::phoenix::cref(AuthMethod), boost::phoenix::cref(AuthInitialResponse))];

        unknown = eps[qi::_val = construct<commands::Unknown>(ctx)];
        syntaxError = eps[qi::_val = construct<commands::SyntaxError>(ctx)];

        start = no_case["EHLO"] >> +blank >> (ehlo | syntaxError)[handler]
            | no_case["LHLO"] >> +blank >> (lhlo | syntaxError)[handler]
            | no_case["HELO"] >> +blank >> (helo | syntaxError)[handler]
            | no_case["MAIL "] >> (mailfrom | syntaxError)[handler]
            | no_case["RCPT "] >> (rcptto | syntaxError)[handler]
            | (data | quit | rset | startTls | noop | auth | unknown)[handler];
    }

    qi::rule<Iterator, std::string()> esmtp_key, esmtp_value, path, ehlo_host;
    qi::rule<Iterator, commands::Params()> esmtp_params;
    qi::rule<Iterator, std::pair<std::string, std::string>()> esmtp_pair;
    qi::rule<Iterator, commands::Ehlo()> ehlo;
    qi::rule<Iterator, commands::Lhlo()> lhlo;
    qi::rule<Iterator, commands::Helo()> helo;
    qi::rule<Iterator, commands::MailFrom()> mailfrom;
    qi::rule<Iterator, commands::RcptTo()> rcptto;
    qi::rule<Iterator, commands::Data()> data;
    qi::rule<Iterator, commands::Quit()> quit;
    qi::rule<Iterator, commands::Rset()> rset;
    qi::rule<Iterator, commands::StartTls()> startTls;
    qi::rule<Iterator, commands::Noop()> noop;
    qi::rule<Iterator, commands::Auth()> auth;
    qi::rule<Iterator, commands::Unknown()> unknown;
    qi::rule<Iterator, commands::SyntaxError()> syntaxError;
    qi::rule<Iterator> start;

private:
    struct HandlerImpl {
        mutable Handler handler;

        HandlerImpl(Handler handler) : handler(std::move(handler)) {}

        template <typename T>
        struct result { typedef void type; };

        template <typename Command>
        void operator()(Command v) const { handler(v); }
    };

    HandlerImpl handler;
    boost::iterator_range<Iterator> ctx;

    std::string AuthInitialResponse;
    commands::AuthMethod AuthMethod;
};

}   // namespace parser

template <class Iterator, class Handler>
Iterator parse_command(Iterator beg, Iterator end, Handler handler) {
    parser::CommandGrammar<Iterator, Handler> gr(boost::make_iterator_range(beg, end), handler);

    boost::spirit::qi::parse(beg, end, gr);
    return beg;
}

}   // namespace ymod_smtpserver
