#pragma once

#include <memory>

namespace ymod_smtpserver {
namespace parser {

// Find end of message token (^|\n).\r?\n
struct EOMParser {
    template <typename Iterator>
    bool parse(Iterator beg, Iterator end, Iterator& eomBeg, Iterator& eomEnd);

    void reset() { state = LineBegin; }

private:
    enum State {
        LineBegin,     // ^
        InLine,        // *
        Dot,           // ^.
        CrAfterDot     // ^.\r
    };
    State state = LineBegin;
};

// Finds eom token in [beg, end) input range.
/**
 * If the eom token was found:
 * returns true,
 * eomBeg: the beginning of the eom token suffix in [beg, end) input range
 * eomEnd: iterator pointing directly past the last character of the eom token
 *
 * Otherwise:
 * returns false (needs more text)
 * eomBeg: the end of the portion of the parsed text (might include the prefix of the eom token)
 * eomEnd: the end of the text parsed
 *
 * The caller must guarantee that upon the next invocation of this function [eomBeg, end) range
 * should still be valid and 'eomBeg' will point to the place from previous step
 */
template <typename Iterator>
bool EOMParser::parse(Iterator beg, Iterator end, Iterator& eomBeg, Iterator& eomEnd) {
    for (; beg != end; ++beg) {
        switch (state) {

        case InLine:
            if (*beg == '\n') {         // * -> ^
                state = LineBegin;
            }
            break;

        case LineBegin:
            if (*beg == '.') {          // ^ -> ^.
                state = Dot;
                eomBeg = beg;
            } else if (*beg != '\n') {  // ^ -> *
                state = InLine;
            }
            break;

        case Dot:
            if (*beg == '\r') {         // ^. -> ^.\r
                state = CrAfterDot;
            } else if (*beg == '\n') {  // ^. -> ^.\n
                state = LineBegin;
                eomEnd = beg + 1;
                return true;
            } else {                    // ^. -> *
                state = InLine;
            }
            break;

        case CrAfterDot:
            if (*beg == '\n') {         // ^.\r -> ^.\r\n
                state = LineBegin;
                eomEnd = beg + 1;
                return true;
            } else {
                state = InLine;         // ^.\r -> *
            }
            break;
        } // switch
    } // for
    eomEnd = end;
    if (state != Dot && state != CrAfterDot) {
        eomBeg = eomEnd;
    }
    return false;
}


// Find leading dot ^.
struct DotStuffingParser {
    template <typename Iterator>
    bool parse(Iterator beg, Iterator end, Iterator& dot);

    void reset() { state = LineBegin; }

private:
    enum State {
        LineBegin,     // ^
        InLine         // *
    };
    State state = LineBegin;
};


// Finds ^. token in [beg, end) input range.
/**
 * If the token was found:
 * returns true,
 * dot: iterator pointing the '.' symbol
 *
 * Otherwise:
 * returns false (needs more text)
 * dot: the end of the text parsed
 */
template <typename Iterator>
bool DotStuffingParser::parse(Iterator beg, Iterator end, Iterator& dot) {
    for(; beg != end; ++beg) {
        switch (state) {

        case LineBegin:
            if (*beg == '.') {      // ^ -> ^.
                state = InLine;
                dot = beg;
                return true;
            } else if (*beg != '\n') {                // ^ -> *
                state = InLine;
            }
            break;

        case InLine:
            if (*beg == '\n') {     // * -> ^
                state = LineBegin;
            }
            break;
        }
    }
    dot = end;
    return false;
}


struct MessageParser {
    using StringPtr = std::shared_ptr<std::string>;

    explicit MessageParser(bool removeLeadingDots = true)
        : removeLeadingDots(removeLeadingDots)
        , message(std::make_shared<std::string>())
    {}

    template <typename Iterator>
    bool parse(Iterator beg, Iterator end, Iterator& processed);

    void reset() {
        eomShift = 0;
        eomParser.reset();
        dotStuffingParser.reset();
        message = std::make_shared<std::string>();
    }

    StringPtr getMessage() { return message; }

private:
    bool removeLeadingDots = true;
    StringPtr message;
    EOMParser eomParser;
    std::size_t eomShift = 0;
    DotStuffingParser dotStuffingParser;
};

template <typename Iterator>
bool MessageParser::parse(Iterator beg, Iterator end, Iterator& processed) {
    Iterator eom = beg;
    Iterator eomEnd;
    bool foundEOM = eomParser.parse(beg + eomShift, end, eom, eomEnd);
    eomShift = std::distance(eom, eomEnd);

    if (removeLeadingDots) {
        while (beg != eom) {
            Iterator dot;
            if (dotStuffingParser.parse(beg, eom, dot)) {
                message->append(beg, dot);
                beg = ++dot;
            } else {
                message->append(beg, eom);
                beg = eom;
            }
        }
    } else {
        message->append(beg, eom);
    }
    processed = foundEOM ? eomEnd : eom;
    return foundEOM;
}

}   // namespace parser
}   // namespace ymod_smtpserver
