#pragma once

#include <yplatform/net/settings.h>
#include <yplatform/net/ssl_settings.h>
#include <yplatform/time_traits.h>

namespace ymod_smtpserver {

namespace amount {

constexpr std::size_t KB = 1024;
constexpr std::size_t MB = 1024 * KB;

}   // namespace amount

struct Settings {
    virtual ~Settings() {}

    yplatform::net::server_settings socket;
    yplatform::net::ssl_settings ssl;

    struct Endpoint {
        std::string addr;
        unsigned short port = 0;
        bool secure = false;
    };
    std::vector<Endpoint> endpoints;

    struct Timeouts {
        using Duration = yplatform::time_traits::duration;

        Duration readCommand = yplatform::time_traits::seconds(5);
        Duration readMessage = yplatform::time_traits::seconds(30);
        Duration writeResponse = yplatform::time_traits::seconds(5);
    };
    Timeouts timeouts;

    std::size_t chunkSize = amount::KB * 8;
    bool removeLeadingDots = true;
    bool fixCrlf = false;

    virtual void parsePtree(const yplatform::ptree& conf) {
        socket.parse_ptree(conf.get_child("socket", yplatform::ptree()));
        ssl.parse_ptree(conf.get_child("ssl", yplatform::ptree()));

        auto range = conf.get_child("endpoints").equal_range("listen");
        for (auto it = range.first; it != range.second; ++it) {
            const auto & ep = it->second;
            Endpoint point;
            auto addrOpt = ep.get_optional<std::string>("<xmlattr>.addr");
            point.addr = addrOpt ? addrOpt.value() : ep.get<std::string>("addr", "");
            auto portOpt = ep.get_optional<unsigned short>("<xmlattr>.port");
            point.port = portOpt ? portOpt.value() : ep.get<unsigned short>("port", point.port);
            point.secure = ep.get<bool>("ssl", point.secure);
            endpoints.push_back(std::move(point));
        }

        if (auto timeoutsOpt = conf.get_child_optional("timeouts")) {
            const auto& timeoutsConf = timeoutsOpt.get();
            timeouts.readCommand = timeoutsConf.get("read_command", timeouts.readCommand);
            timeouts.readMessage = timeoutsConf.get("read_message", timeouts.readMessage);
            timeouts.writeResponse = timeoutsConf.get("write_response", timeouts.writeResponse);
        }

        chunkSize = conf.get("read_chunk_size", chunkSize);
        removeLeadingDots = conf.get("remove_leading_dots", removeLeadingDots);
        fixCrlf = conf.get("fix_crlf", fixCrlf);
    }
};

using SettingsPtr = std::shared_ptr<Settings>;

}   // namespace ymod_smtpserver
