#pragma once

#include <boost/range/iterator_range.hpp>

#include <string>
#include <memory>
#include <optional>

namespace ymod_smtpserver::NUtil {

/// Finds (\r)*\n{0,1} sequence and returns [sequenceStart, sequenceEnd) or returns nullopt if the sequence wasn't found
template<typename Iter>
std::optional<boost::iterator_range<Iter>> FindPossiblyIncorrectCrlfSequence(Iter beg, Iter end) {
    enum class EState {
        Initial = 0,
        AfterCR
    };

    auto it = beg;
    EState state = EState::Initial;

    Iter sequenceBegin;
    while (it != end) {
        switch (state) {
            case EState::Initial:
                if (*it == '\n') {
                    return boost::iterator_range<Iter>{it, it + 1};
                } else if (*it == '\r') {
                    state = EState::AfterCR;
                    sequenceBegin = it;
                } else {
                    // do nothing
                }
                break;
            case EState::AfterCR:
                if (*it == '\n') {
                    return boost::iterator_range<Iter>{sequenceBegin, it + 1};
                } else if (*it == '\r') {
                    // do nothing
                } else {
                    return boost::iterator_range<Iter>{sequenceBegin, it};
                }
                break;
        }
        ++it;
    }

    // it == end
    if (state == EState::AfterCR) {
        return boost::iterator_range<Iter>{sequenceBegin, it};
    }

    return {};
}

template<typename Iter>
std::shared_ptr<std::string> FixCrlf(Iter begin, Iter end) {
    auto result = std::make_shared<std::string>();
    result->reserve(std::distance(begin, end));

    auto it = begin;
    while (it != end) {
        auto badSequence = FindPossiblyIncorrectCrlfSequence(it, end);

        if (badSequence) {
            result->append(it, badSequence->begin());
            result->append("\r\n");
            it = badSequence->end();
        } else {
            result->append(it, end);
            it = end;
        }
    }

    return result;
}

}
