#include "impl.h"
#include <pa/async.h>
#include <yplatform/find.h>

namespace ymod_stat_server {

using std::string;

void impl::init(const yplatform::ptree& xml)
{
    settings_.parse_ptree(xml);

    if (settings_.pa_log_file.size())
    {
        YLOG_LOCAL(info) << "init pa, path: " << settings_.pa_log_file;
        pa::async_profiler::init(
            settings_.pa_max_size, settings_.pa_dump_size, settings_.pa_log_file);
    }

    auto reactor = yplatform::find_reactor(xml.get("reactor", "global"));
    io_data_.reset(new yplatform::net::io_data(*reactor->io()));

    if (settings_.trace_reactors)
    {
        clock_generator_.reset(new yplatform::time_traits::timer(*io_data_->get_io()));
    }

    tcp_server_.reset(new yplatform::net::tcp_server(
        *io_data_, settings_.listen_addr, settings_.listen_port, settings_.socket));
}

void impl::reload(const yplatform::ptree& xml)
{
    settings new_settings;
    new_settings.parse_ptree(xml);
    std::lock_guard<decltype(spinlock_)> lock(spinlock_);
    settings_ = new_settings;
}

void impl::start(void)
{
    // Where is no need to capture shared from this because of
    // yplatform modules and reactor destroy order guarantees.
    tcp_server_->listen([this](yplatform::net::tcp_socket&& socket) mutable {
        settings st;
        {
            std::lock_guard<decltype(spinlock_)> lock(spinlock_);
            st = settings_;
        }
        auto sess = boost::make_shared<session>(std::move(socket), settings_);
        sess->start();
    }); // listen()

    if (clock_generator_) async_wait_clock_generator();
}

void impl::stop()
{
    if (clock_generator_) clock_generator_->cancel();
    tcp_server_->stop();
}

void impl::handle_trace_reactor(
    const std::string& name,
    const std::size_t index,
    const time_point& start)
{
    auto time_delta = yplatform::time_traits::clock::now() - start;
    stats_->add_trace_result(
        name + std::to_string(index),
        yplatform::time_traits::duration_cast<milliseconds>(time_delta));
}

void impl::trace_reactors()
{
    auto all_reactors = yplatform::global_reactor_set->get_all();
    for (auto const& pair : all_reactors)
    {
        for (std::size_t i = 0; i < pair.second->size(); ++i)
        {
            (*pair.second)[i]->io()->post(boost::bind(
                &impl::handle_trace_reactor,
                this,
                pair.first,
                i,
                yplatform::time_traits::clock::now()));
        }
    }
}

void impl::async_wait_clock_generator()
{
    clock_generator_->expires_from_now(yplatform::time_traits::seconds(1));
    clock_generator_->async_wait(boost::bind(&impl::handle_clock_generator, this, _1));
}

void impl::handle_clock_generator(const boost::system::error_code& ec)
{
    if (ec != boost::asio::error::operation_aborted)
    {
        trace_reactors();
        async_wait_clock_generator();
    }
}

}

namespace ymod_statserver = ymod_stat_server;

#include <yplatform/module_registration.h>
REGISTER_MODULE(ymod_stat_server::impl)
