#pragma once

#include "session.h"
#include "uptime.h"
#include <yplatform/net/server.h>
#include <yplatform/module.h>
#include <yplatform/spinlock.h>
#include <yplatform/time_traits.h>
#include <yplatform/util/per_second_accumulator.h>

namespace ymod_stat_server {

class stats : public yplatform::module_stats
{
    using milliseconds = yplatform::time_traits::milliseconds;

    struct trace_results
    {
        trace_results() : acc(61){};
        milliseconds recent_trace;
        mutable yplatform::per_second_accumulator<milliseconds> acc;
    };

public:
    void add_trace_result(const std::string& name, const milliseconds& ms)
    {
        lock_t lock(mux());
        auto& results = trace_results_[name];
        results.recent_trace = ms;
        results.acc.add(ms);
    }

    yplatform::ptree get_stats() const
    {
        yplatform::ptree ret;
        ret.put("just_started", uptime() < boost::posix_time::minutes(1) ? 1 : 0);

        lock_t lock(mux());
        for (auto const& pair : trace_results_)
        {
            auto& name = pair.first;
            auto& results = pair.second;
            auto latency = results.acc.get_max(60);
            ret.put(
                std::string("reactors." + name + "-trace"),
                std::to_string(results.recent_trace.count()));
            ret.put(
                std::string("max_latency_in_last_minute." + name), std::to_string(latency.count()));
        }
        return ret;
    }

private:
    std::map<std::string, trace_results> trace_results_;
};

typedef boost::shared_ptr<stats> stats_ptr;

class impl : public yplatform::module
{
public:
    void init(const yplatform::ptree& xml);
    void reload(const yplatform::ptree& xml);
    void start();
    void stop(); // TODO make fini

    yplatform::ptree get_stats() const
    {
        return stats_->get_stats();
    }

private:
    using time_point = yplatform::time_traits::time_point;
    using milliseconds = yplatform::time_traits::milliseconds;

    void handle_trace_reactor(
        const std::string& name,
        const std::size_t index,
        const time_point& start);
    void trace_reactors();
    void async_wait_clock_generator();
    void handle_clock_generator(const boost::system::error_code& ec);

private:
    settings settings_;
    std::unique_ptr<yplatform::time_traits::timer> clock_generator_;
    std::unique_ptr<yplatform::net::io_data> io_data_;
    std::unique_ptr<yplatform::net::tcp_server> tcp_server_;
    yplatform::spinlock spinlock_;
    stats_ptr stats_ = boost::make_shared<stats>();
};

}
