#include <unistd.h>
#include <cstdio>
#include <cstdlib>
#include <cctype>
#include <cstring>
#include <algorithm>

#include <boost/property_tree/json_parser.hpp>

#include <yplatform/repository.h>
#include <yplatform/context_repository.h>
#include <yplatform/util/split.h>

#include <boost/algorithm/string/join.hpp>

#include "session.h"
#include "uptime.h"

namespace ymod_stat_server {

using std::string;

const string INDENT_STR = "    ";

template <typename Range>
std::basic_string<typename ::boost::range_value<Range>::type> xml_encode(Range const& src)
{
    std::basic_string<typename ::boost::range_value<Range>::type> ret;
    ret.reserve(src.size() * 10 / 8);

    for (typename ::boost::range_const_iterator<Range>::type i = src.begin(); i != src.end(); ++i)
    {
        switch (*i)
        {
        default:
            ret += *i;
            break;
        case '"':
            ret += "&quot;";
            break;
        case '&':
            ret += "&amp;";
            break;
        case '\'':
            ret += "&apos;";
            break;
        case '<':
            ret += "&lt;";
            break;
        case '>':
            ret += "&gt;";
            break;
        }
    }

    return ret;
}

class xml_tag_cleaner
{
public:
    xml_tag_cleaner(const std::string& tag_name) : tag_name(tag_name)
    {
    }

    std::ostream& apply(std::ostream& stream) const
    {
        if (tag_name.empty())
        {
            stream << "_empty";
            return stream;
        }

        numeric_prefix(stream);
        for (const char symbol : tag_name)
        {
            stream << replace_special_symbol(symbol);
        }
        return stream;
    }

private:
    const std::string& tag_name;

    static char replace_special_symbol(char value)
    {
        switch (value)
        {
        case '/':
        case ':':
            return '_';
        default:
            return value;
        }
    }

    void numeric_prefix(std::ostream& stream) const
    {
        if (std::isdigit(*tag_name.begin()))
        {
            stream << '_';
        }
    }
};

std::ostream& operator<<(std::ostream& stream, const xml_tag_cleaner& cleaner)
{
    return cleaner.apply(stream);
}

xml_tag_cleaner xml_tag_clean(const std::string& tag_name)
{
    return xml_tag_cleaner(tag_name);
}

const std::string stat_node_name = "stat";
const std::string sessions_node_name = "sessions";
const std::string modules_node_name = "modules";
const std::string uptime_node_name = "uptime";

request make_request(const std::string& path)
{
    if (path.empty())
    {
        return request();
    }

    request result;
    auto parsed = yplatform::util::split(path, ".");
    if (parsed[0] != stat_node_name)
    {
        result.dump_stat = false;
        return result;
    }

    if (parsed.size() < 2)
    {
        return result;
    }

    auto& node_name = parsed[1];
    if (node_name == sessions_node_name)
    {
        result.dump_modules = false;
        result.dump_uptime = false;
    }
    else if (node_name == modules_node_name)
    {
        result.dump_sessions = false;
        result.dump_uptime = false;
        if (parsed.size() > 2)
        {
            result.module_name = parsed[2];
        }

        if (parsed.size() > 3)
        {
            result.module_path = boost::algorithm::join(
                std::vector<std::string>(parsed.begin() + 3, parsed.end()), ".");
        }
    }
    else if (node_name == uptime_node_name)
    {
        result.dump_sessions = false;
        result.dump_modules = false;
    }
    else
    {
        result.dump_sessions = false;
        result.dump_modules = false;
        result.dump_uptime = false;
    }
    return result;
}

boost::optional<yplatform::ptree> reduce_module_stats(
    const yplatform::ptree& module_stats,
    const std::string& path)
{
    yplatform::ptree result;
    auto child_opt = module_stats.get_child_optional(path);
    if (child_opt)
    {
        result.put_child(path, *child_opt);
        return result;
    }
    auto value_opt = module_stats.get_optional<std::string>(path);
    if (value_opt)
    {
        result.put(path, *value_opt);
        return result;
    }
    return boost::optional<yplatform::ptree>();
}

session::session(tcp_socket&& socket, const settings& settings)
    : socket_(std::move(socket))
    , settings_(settings)
    , write_buffer_(std::numeric_limits<std::size_t>::max())
    , read_buffer_(settings.path_max_size)
{
    YLOG_L(debug) << "new stat session " << socket_.id();
}

session::~session()
{
    if (socket_.is_open())
        YLOG_L(debug) << "closing stat session " << socket_.id() << " with "
                      << socket_.remote_addr().to_string() << ':' << socket_.remote_port();
}

void session::start()
{
    socket_.async_read(
        read_buffer_,
        settings_.path_read_timeout,
        [this,
         self = shared_from_this()](const boost::system::error_code& /*e*/, std::size_t /*bytes*/) {
            auto path = std::string(
                std::istreambuf_iterator<char>(&read_buffer_), std::istreambuf_iterator<char>());
            auto req = make_request(path);
            respond(req);
        });
}

void session::respond(const request& req)
{
    std::ostream stream(&write_buffer_);

    if (settings_.dump_json)
    {
        const auto data = make_response_data(req);
        boost::property_tree::write_json(stream, data);
    }
    else
    {
        write_xml(stream);
    }

    socket_.async_write(
        write_buffer_,
        yplatform::time_traits::duration::max(),
        boost::bind(&session::handle_quit, this->shared_from_this(), _1, _2));
}

void session::write_xml(std::ostream& stream) const
{
    using lock_t = boost::unique_lock<boost::mutex>;

    stream << "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n";
    stream << "<stat>\n";

    if (settings_.dump_sessions)
    {
        stream << "<sessions>\n";
        auto contexts = yplatform::context_repository::instance().get_contexts();
        for (yplatform::task_context_list::const_iterator i = contexts->begin(),
                                                          i_end = contexts->end();
             i != i_end;
             ++i)
        {
            stream << INDENT_STR << "<" << xml_tag_clean((*i)->get_name()) << ">\n";
            stream << INDENT_STR << INDENT_STR << "<uniq_id>" << (*i)->uniq_id() << "</uniq_id>\n";
            yplatform::task_context::ptree_ptr stat = (*i)->get_stat();
            ptree_to_xml(*stat, stream, 2);
            stream << INDENT_STR << "</" << xml_tag_clean((*i)->get_name()) << ">\n";
        }
        stream << "</sessions>\n";
    }

    {
        stream << "<modules>\n";
        lock_t lock(yplatform::repository::instance().mux());
        for (yplatform::repository::map_t::const_iterator
                 i = yplatform::repository::instance().get_services().begin(),
                 i_end = yplatform::repository::instance().get_services().end();
             i != i_end;
             ++i)
        {
            stream << INDENT_STR << "<" << xml_tag_clean(i->first) << ">\n";
            auto module = i->second.get_module_safe();
            yplatform::module_stats_ptr stat = module->get_module_stats();
            if (stat)
            {
                ptree_to_xml(*stat->get_stat(), stream, 2);
            }
            else
            {
                yplatform::ptree stats_ptree = module->get_stats();
                if (stats_ptree.size() || stats_ptree.data().size())
                {
                    ptree_to_xml(stats_ptree, stream, 2);
                }
            }
            stream << INDENT_STR << "</" << xml_tag_clean(i->first) << ">\n";
        }
        stream << "</modules>\n";
    }
    stream << "<uptime>\n"
           << INDENT_STR << uptime() << "\n"
           << "</uptime>\n";
    stream << "</stat>\n";
}

void session::handle_quit(const boost::system::error_code& /*e*/, std::size_t /*bytes*/)
{
}

void session::ptree_to_xml(const yplatform::ptree& data, std::ostream& stream, int indent)
{
    for (yplatform::ptree::const_iterator i = data.begin(), i_end = data.end(); i != i_end; ++i)
    {
        for (int j = 0; j < indent; ++j)
            stream << INDENT_STR;

        if (i->second.empty())
        {
            stream << "<" << xml_tag_clean(i->first) << ">" << xml_encode(i->second.data()) << "</"
                   << xml_tag_clean(i->first) << ">\n";
            continue;
        }
        stream << "<" << xml_tag_clean(i->first) << ">\n";

        ptree_to_xml(i->second, stream, indent + 1);

        for (int j = 0; j < indent; ++j)
            stream << INDENT_STR;
        stream << "</" << xml_tag_clean(i->first) << ">\n";
    }
}

yplatform::ptree session::make_response_data(const request& req) const
{
    using yplatform::ptree;
    using lock_t = boost::unique_lock<boost::mutex>;

    ptree stat;
    if (!req.dump_stat)
    {
        return stat;
    }

    if (settings_.dump_sessions && req.dump_sessions)
    {
        ptree sessions;
        auto contexts = yplatform::context_repository::instance().get_contexts();
        for (const auto& context : *contexts)
        {
            ptree session;
            session.put("uniq_id", context->uniq_id());
            session.push_back({ "stat", *context->get_stat() });
            sessions.push_back({ context->get_name(), std::move(session) });
        }
        if (!sessions.empty())
        {
            stat.push_back({ sessions_node_name, std::move(sessions) });
        }
    }

    if (req.dump_modules)
    {
        ptree modules;
        lock_t lock(yplatform::repository::instance().mux());
        for (const auto& service : yplatform::repository::instance().get_services())
        {
            ptree module_stats;
            auto module = service.second.get_module_safe();
            if (req.module_name.size() && req.module_name != module->name())
            {
                continue;
            }
            if (const auto stats = module->get_module_stats())
            {
                module_stats = *stats->get_stat();
            }
            else
            {
                module_stats = module->get_stats();
            }

            if (req.module_path.empty())
            {
                modules.push_back({ service.first, std::move(module_stats) });
                continue;
            }

            auto reduced_opt = reduce_module_stats(module_stats, req.module_path);
            if (reduced_opt)
            {
                modules.push_back({ service.first, std::move(*reduced_opt) });
            }
        }
        if (!modules.empty())
        {
            stat.push_back({ modules_node_name, std::move(modules) });
        }
    }

    if (req.dump_uptime)
    {
        std::ostringstream ss;
        ss << uptime();
        stat.put(uptime_node_name, ss.str());
    }

    ptree result;
    if (!stat.empty())
    {
        result.push_back({ stat_node_name, std::move(stat) });
    }
    return result;
}

}
