#pragma once

#include <library/c/tvmauth/tvmauth.h>
#include <boost/system/error_code.hpp>
#include <string>
#include <vector>

namespace ymod_tvm { namespace error {

enum code
{
    success = 0,
    tickets_not_loaded,
    keys_not_loaded,
    unknown_service,
    no_ticket_for_service,
    context_is_not_ready,
    unknown_error,

    COUNT
};
}}

namespace boost { namespace system {

template <>
struct is_error_code_enum<ymod_tvm::error::code>
{
    static constexpr bool value = true;
};

template <>
struct is_error_code_enum<TA_EErrorCode>
{
    static constexpr bool value = true;
};

} // namespace system
} // namespace boost

namespace ymod_tvm {

using boost::system::error_code;

class error_category : public boost::system::error_category
{
public:
    const char* name() const noexcept override
    {
        return "ymod_tvm";
    }

    std::string message(int ev) const override
    {
        static constexpr std::initializer_list<const char*> names = {
            "success",
            "tvm tickets have not been loaded",
            "tvm keys have not been loaded",
            "unknown service",
            "no ticket available for given service",
            "context is not ready",
            "unknown error"
        };
        static_assert(
            names.size() == static_cast<size_t>(error::COUNT),
            "Error codes count doesn't correspond with error names count");
        static const std::vector<std::string> messages(names.begin(), names.end());

        return messages[ev];
    }

    static const error_category& instance()
    {
        static const error_category category;
        return category;
    }

private:
    error_category() = default;
};

class libticketparser2_error_category : public boost::system::error_category
{
public:
    const char* name() const noexcept override
    {
        return "libticket-parser2";
    }

    std::string message(int ev) const override
    {
        return TA_ErrorCodeToString(static_cast<TA_EErrorCode>(ev));
    }

    static const libticketparser2_error_category& instance()
    {
        static const libticketparser2_error_category category;
        return category;
    }

private:
    libticketparser2_error_category() = default;
};

namespace error {
inline error_code make_error_code(error::code err = error::success)
{
    return error_code(err, error_category::instance());
}
}
} // namespace ymod_tvm

// Has to be in the global namespace for argument-dependent lookup.
inline ymod_tvm::error_code make_error_code(TA_EErrorCode err)
{
    return ymod_tvm::error_code(err, ymod_tvm::libticketparser2_error_category::instance());
}
