#pragma once

#include <ymod_tvm/error.h>
#include <ymod_tvm/types.h>
#include <yplatform/task_context.h>
#include <boost/optional.hpp>
#include <boost/variant.hpp>
#include <functional>
#include <string>

namespace ymod_tvm {

using yplatform::task_context;
using yplatform::task_context_ptr;

class tvm2_module
{
public:
    using tickets_ready_callback = std::function<
        void(const error_code& error, const std::string& service, const std::string& ticket)>;

    using callback = std::function<void()>;

    // Another modules subscribe to ticket updates. Tickets are not available
    // at application start so callers will receive them later.
    virtual void subscribe_service_ticket(
        const std::string& service,
        const tickets_ready_callback& cbs) = 0;

    // The callback will be invoked after tickets were loaded first time.
    virtual void subscribe_all_tickets_are_ready(const callback& cbs) = 0;

    // Check service ticket.
    virtual boost::system::error_code check_service_ticket(
        task_context_ptr ctx,
        const std::string& ticket) = 0;

    // Get native service ticket.
    // Has to return boost::optional as service_ticket can't be dafault-constructed.
    virtual boost::optional<tvm2::service_ticket> get_native_service_ticket(
        task_context_ptr ctx,
        const std::string& ticket) = 0;

    // Get native service ticket.
    // Do not suppress any errors and exceptions.
    virtual boost::variant<error_code, tvm2::service_ticket> get_native_service_ticket_or_error(
        const std::string& ticket) = 0;

    // Unreliable ticket accessor – will return error if can't acquire the ticket.
    virtual boost::system::error_code get_service_ticket(
        const std::string& service,
        std::string& ticket) = 0;

    // Check user ticket.
    virtual boost::system::error_code check_user_ticket(
        task_context_ptr ctx,
        tvm2::blackbox_env env,
        const std::string& ticket) = 0;

    // Get native user ticket.
    // Has to return boost::optional as user_ticket can't be dafault-constructed.
    virtual boost::optional<tvm2::user_ticket> get_native_user_ticket(
        task_context_ptr ctx,
        tvm2::blackbox_env env,
        const std::string& ticket) = 0;

    // Get native user ticket.
    // Do not suppress any errors and exceptions.
    virtual boost::variant<error_code, tvm2::user_ticket> get_native_user_ticket_or_error(
        tvm2::blackbox_env env,
        const std::string& ticket) = 0;

    // Callback will be invoked when tvm keys are loaded (module is ready to check tickets).
    virtual void subscribe_keys_loaded(const callback& cb) = 0;

    virtual ~tvm2_module() = default;
};

}
