#include "impl.h"

#include "read_file.h"
#include <ymod_tvm/error.h>
#include <yplatform/find.h>
#include <yplatform/util/safe_call.h>
#include <ymod_httpclient/client.h>

#include <boost/property_tree/json_parser.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <algorithm>
#include <sstream>
#include <future>

namespace ymod_tvm { namespace tvm2 {
namespace {
template <typename Callbacks, typename... Args>
void invoke_all(yplatform::reactor& reactor, const Callbacks& cbs, Args&&... args)
{
    for (auto& cb : cbs)
    {
        reactor.io()->post(
            [cb, args...]() mutable { yplatform::safe_call(cb, std::forward<Args>(args)...); });
    }
}
}

namespace p = std::placeholders;

impl::impl(yplatform::reactor& reactor, const yplatform::ptree& config)
    : impl(reactor, settings(config))
{
}

impl::impl(yplatform::reactor& reactor, settings s)
    : settings_(std::move(s))
    , reactor_(reactor)
    , keys_timer_(std::make_shared<timer>(*reactor_.io()))
    , tickets_timer_(std::make_shared<timer>(*reactor_.io()))
    , http_client_(std::make_shared<yhttp::cluster_client>(reactor_, settings_.http))
{
    libtvm_version_ = NTvmAuthWrapper::LibVersion();
    http_client_->logger(logger());

    if (settings_.tvm_secret.size())
    {
        service_sign_context_ = std::make_shared<service_context>(
            service_context::SigningFactory(settings_.tvm_secret, settings_.my_tvm_id));
    }
    ctx_ = boost::make_shared<task_context>();
    if (settings_.wait_first_update_on_start)
    {
        settings start_settings = settings_;
        start_settings.wait_first_update_on_start = false;
        tvm start_tvm(start_settings);
        auto keys_prom = std::make_shared<std::promise<void>>();
        auto tickets_prom = std::make_shared<std::promise<void>>();
        start_tvm.subscribe_keys_loaded([keys_prom]() { keys_prom->set_value(); });
        start_tvm.subscribe_all_tickets_are_ready([tickets_prom]() { tickets_prom->set_value(); });
        start_tvm.start();
        keys_prom->get_future().wait();
        tickets_prom->get_future().wait();
        state_ = start_tvm.impl_->state();
    }
}

void impl::start()
{
    started_ = true;
    if (settings_.wait_first_update_on_start)
    {
        // If keys and tickets were loaded beforehand, notify clients on start.
        invoke_ticket_callbacks();
        invoke_all(reactor_, keys_ready_callbacks_);
        keys_ready_callbacks_.clear();
        schedule_update_keys(settings_.keys_update_interval);
        schedule_update_tickets(settings_.tickets_update_interval);
    }
    else
    {
        update_keys();
        update_tickets();
    }
}

void impl::stop()
{
    guard_t guard(lock_);
    if (keys_timer_) keys_timer_->cancel();
    if (tickets_timer_) tickets_timer_->cancel();
}

void impl::schedule_update_keys(duration interval)
{
    guard_t guard(lock_);
    keys_timer_->expires_from_now(interval);
    keys_timer_->async_wait(std::bind(&impl::update_keys, shared_from(this), p::_1));
}

void impl::update_keys(const error_code& err)
{
    // Operation cancelled.
    if (err) return;

    if (settings_.keys_file.size())
    {
        update_keys_from_file();
    }
    else
    {
        update_keys_from_api();
    }
}

void impl::update_keys_from_file()
{
    try
    {
        // Try only once, because keys never should be read from file
        // in a production environment.
        if (load_keys(read_file(settings_.keys_file)))
        {
            YLOG_L(warning) << "tvm keys loaded from file " << settings_.keys_file;
            invoke_all(reactor_, keys_ready_callbacks_);
            keys_ready_callbacks_.clear();
        }
    }
    catch (const std::exception& e)
    {
        YLOG_L(error) << "tvm keys read from file failed: " << e.what();
    }
}

void impl::update_keys_from_api()
{
    http_client_->async_run(
        ctx_,
        yhttp::request::GET(
            settings_.keys_request + yhttp::url_encode({ { "lib_version", libtvm_version_ } })),
        [this, self = shared_from(this)](const error_code& err, yhttp::response resp) {
            auto next_interval = settings_.retry_interval;
            if (err || resp.status != 200)
            {
                YLOG_L(error) << "failed to load tvm keys: "
                              << (err ? err.message() :
                                        std::to_string(resp.status) + " " + resp.body);
            }
            else if (load_keys(resp.body))
            {
                next_interval = settings_.keys_update_interval;
                // Notify clients first keys are loaded.
                invoke_all(reactor_, keys_ready_callbacks_);
                keys_ready_callbacks_.clear();
            }
            schedule_update_keys(next_interval);
        });
}

bool impl::load_keys(const string& body)
{
    try
    {
        auto new_check_context = std::make_shared<service_context>(
            service_context::CheckingFactory(settings_.my_tvm_id, body));
        service_check_context(new_check_context);
        auto new_user_contexts = std::make_shared<std::map<blackbox_env, user_context>>();
        for (auto env : settings_.blackbox_envs)
        {
            // In case of failure creating user context for one env,
            // still try to create others.
            yplatform::safe_call("create user context", [&new_user_contexts, &body, env]() {
                new_user_contexts->insert(std::make_pair(env, user_context(env, body)));
            });
        }
        user_contexts(new_user_contexts);
        YLOG_L(info) << "tvm keys loaded";
    }
    catch (const std::exception& e)
    {
        YLOG_L(error) << "tvm keys load failed: " << e.what();
        return false;
    }
    return true;
}

void impl::schedule_update_tickets(duration interval)
{
    guard_t guard(lock_);
    tickets_timer_->expires_from_now(interval);
    tickets_timer_->async_wait(std::bind(&impl::update_tickets, shared_from(this), p::_1));
}

void impl::update_tickets(const error_code& err)
{
    // Operation cancelled.
    if (err) return;
    // Don't update tickets if no secret is available (no sign context)
    // or if there are no target services. Invoke ticket callbacks.
    if (!service_sign_context_ || settings_.target_services.empty())
    {
        invoke_ticket_callbacks();
        return;
    }

    string sign;
    auto ts = std::to_string(std::time(nullptr));
    try
    {
        sign = service_sign_context_->SignCgiParamsForTvm(ts, settings_.target_services, "");
    }
    catch (const std::exception& e)
    {
        sign.clear();
        YLOG_L(error) << "failed to update tvm tickets: " << e.what();
    }
    // If signing the request have failed, schedule a retry.
    if (sign.empty())
    {
        schedule_update_tickets(settings_.retry_interval);
        return;
    }

    http_client_->async_run(
        ctx_,
        yhttp::request::POST(
            settings_.tickets_request,
            yhttp::form_encode({ { "grant_type", "client_credentials" },
                                 { "src", settings_.my_tvm_id },
                                 { "dst", settings_.target_services },
                                 { "ts", ts },
                                 { "sign", sign } })),
        [this, self = shared_from(this)](const error_code& err, yhttp::response resp) {
            auto next_interval = settings_.retry_interval;
            if (err || resp.status != 200)
            {
                YLOG_L(error) << "failed to update tvm tickets: "
                              << (err ? err.message() :
                                        std::to_string(resp.status) + " " + resp.body);
            }
            else
            {
                if (parse_tickets(resp.body))
                {
                    next_interval = settings_.tickets_update_interval;
                    invoke_ticket_callbacks();
                }
            }
            schedule_update_tickets(next_interval);
        });
}

bool impl::parse_tickets(const string& resp_body)
{
    bool tickets_loaded = false;
    try
    {
        std::stringstream resp_stream(resp_body);
        boost::property_tree::ptree tickets_json;
        boost::property_tree::json_parser::read_json(resp_stream, tickets_json);
        auto new_tickets = std::make_shared<std::map<string, string>>();
        for (auto& service : settings_.target_services_by_id)
        {
            const auto& id = service.first;
            const auto& name = service.second;
            auto error = tickets_json.get(id + ".error", string());
            auto value = tickets_json.get(id + ".ticket", string());
            if (error.size() || value.empty())
            {
                YLOG_L(error) << "failed to load ticket for " << name << ": "
                              << (error.size() ? error : "missing");
                // Empty ticket value overwrites one (possibly) stored
                // if the response is valid, as TVM API returns error for reason.
            }
            else
            {
                YLOG_L(info) << "loaded ticket for " << name;
            }
            new_tickets->insert(std::make_pair(name, value));
        }
        // Overwrite stored tickets with new values only after request to TVM API is made
        // and valid response is acquired.
        tickets_by_service_name(new_tickets);
        tickets_loaded = true;
        YLOG_L(info) << "tickets updated";
    }
    catch (const std::exception& e)
    {
        YLOG_L(error) << "tvm keys load failed: " << e.what();
    }
    return tickets_loaded;
}

void impl::subscribe_service_ticket(const string& service, const tvm2_callback& cb)
{
    check_not_started();
    callbacks_[service].push_back(cb);
}

void impl::subscribe_all_tickets_are_ready(const callback& cbs)
{
    check_not_started();
    all_tickets_ready_callbacks_.push_back(cbs);
}

void impl::subscribe_keys_loaded(const callback& cb)
{
    check_not_started();
    keys_ready_callbacks_.push_back(cb);
}

error_code impl::check_service_ticket(task_context_ptr ctx, const string& ticket)
{
    auto check_context = service_check_context();
    if (!check_context)
    {
        YLOG_CTX_LOCAL(ctx, error) << "ticket check failed: no keys loaded";
        return error::keys_not_loaded;
    }
    try
    {
        auto ticket_obj = check_context->Check(ticket);
        error_code result = ticket_obj.GetStatus();
        if (ticket_obj)
        {
            YLOG_CTX_LOCAL(ctx, info)
                << "ticket checked successfully, src service is " << ticket_obj.GetSrc();
        }
        else
        {
            YLOG_CTX_LOCAL(ctx, info) << "ticket check failed: " << result.message();
        }
        return result;
    }
    catch (const std::exception& e)
    {
        YLOG_CTX_LOCAL(ctx, error) << "error checking tvm ticket: " << e.what();
        return error::unknown_error;
    }
}

boost::optional<service_ticket> impl::get_native_service_ticket(
    task_context_ptr ctx,
    const string& ticket)
{
    if (auto check_context = service_check_context())
    {
        try
        {
            auto ticket_obj = check_context->Check(ticket);
            if (settings_.log_debug_info)
            {
                yplatform::safe_call([this, &ctx, &ticket_obj]() {
                    YLOG_CTX_LOCAL(ctx, info) << "ticket debug info: " << ticket_obj.DebugInfo();
                });
            }
            return std::move(ticket_obj);
        }
        catch (const std::exception& e)
        {
            YLOG_CTX_LOCAL(ctx, error) << "error getting tvm ticket: " << e.what();
        }
    }
    else
    {
        YLOG_CTX_LOCAL(ctx, error) << "ticket get failed: no keys loaded";
    }
    return boost::none;
}

boost::variant<error_code, service_ticket> impl::get_native_service_ticket_or_error(
    const string& ticket)
{
    const auto check_context = service_check_context();
    if (!check_context)
    {
        return error::keys_not_loaded;
    }
    return check_context->Check(ticket);
}

error_code impl::check_user_ticket(task_context_ptr ctx, blackbox_env env, const string& ticket)
{
    using boost::algorithm::join;
    using boost::adaptors::transformed;
    using NTvmAuthWrapper::TUid;

    auto check_contexts = user_contexts();
    if (!check_contexts)
    {
        YLOG_CTX_LOCAL(ctx, error) << "ticket check failed: no keys loaded";
        return error::keys_not_loaded;
    }
    try
    {
        auto it = check_contexts->find(env);
        if (it == check_contexts->end())
        {
            YLOG_CTX_LOCAL(ctx, error) << "ticket check failed: "
                                          "check context is not ready";
            return error::context_is_not_ready;
        }
        auto& context = it->second;

        auto ticket_obj = context.Check(ticket);
        error_code result = ticket_obj.GetStatus();
        if (ticket_obj)
        {
            string uids_string;
            yplatform::safe_call([&uids_string, &ticket_obj]() {
                auto ticket_uids = ticket_obj.GetUids();
                uids_string = join(
                    ticket_uids | transformed(static_cast<string (*)(TUid)>(&std::to_string)),
                    ", ");
            });
            YLOG_CTX_LOCAL(ctx, info)
                << "user ticket checked successfully, uids: [" << uids_string << "]";
        }
        else
        {
            YLOG_CTX_LOCAL(ctx, info) << "user ticket check failed: " << result.message();
        }
        return result;
    }
    catch (const std::exception& e)
    {
        YLOG_CTX_LOCAL(ctx, error) << "error checking tvm ticket: " << e.what();
        return error::unknown_error;
    }
}

boost::optional<user_ticket> impl::get_native_user_ticket(
    task_context_ptr ctx,
    blackbox_env env,
    const string& ticket)
{
    if (auto check_contexts = user_contexts())
    {
        try
        {
            auto it = check_contexts->find(env);
            if (it == check_contexts->end())
            {
                YLOG_CTX_LOCAL(ctx, error) << "ticket get failed: context is not ready";
            }
            else
            {
                auto& context = it->second;
                auto ticket_obj = context.Check(ticket);
                if (settings_.log_debug_info)
                {
                    yplatform::safe_call([this, &ctx, &ticket_obj]() {
                        YLOG_CTX_LOCAL(ctx, info)
                            << "user ticket debug info: " << ticket_obj.DebugInfo();
                    });
                }
                return std::move(ticket_obj);
            }
        }
        catch (const std::exception& e)
        {
            YLOG_CTX_LOCAL(ctx, error) << "error checking tvm ticket: " << e.what();
        }
    }
    else
    {
        YLOG_CTX_LOCAL(ctx, error) << "ticket get failed: no keys loaded";
    }
    return boost::none;
}

boost::variant<error_code, user_ticket> impl::get_native_user_ticket_or_error(
    blackbox_env env,
    const string& ticket)
{
    const auto check_contexts = user_contexts();
    if (!check_contexts)
    {
        return error::keys_not_loaded;
    }
    const auto it = check_contexts->find(env);
    if (it == check_contexts->end())
    {
        return error::context_is_not_ready;
    }
    const auto& context = it->second;
    return context.Check(ticket);
}

void impl::invoke_ticket_callbacks()
{
    // Invoking tvm2 callbacks.
    for (auto& [service, callbacks] : callbacks_)
    {
        string ticket;
        auto ec = get_service_ticket(service, ticket);
        invoke_all(reactor_, callbacks, ec, service, ticket);
    }

    invoke_all(reactor_, all_tickets_ready_callbacks_);
    all_tickets_ready_callbacks_.clear();
}

error_code impl::get_service_ticket(const std::string& service, std::string& ticket)
{
    auto tickets = tickets_by_service_name();
    if (!tickets) return error::tickets_not_loaded;

    auto it = tickets->find(service);
    if (it == tickets->end())
    {
        return error::unknown_service;
    }
    else if (it->second.empty())
    {
        return error::no_ticket_for_service;
    }

    ticket = it->second;
    return error::success;
}

std::tuple<error_code, std::string> impl::get_service_ticket_for_host(const std::string& host)
{
    auto& service_by_host = settings_.tvm_service_by_host;
    auto it = service_by_host.find(host);
    if (it == service_by_host.end())
    {
        return { error::success, std::string{} };
    }
    std::string ticket;
    auto ec = get_service_ticket(it->second, ticket);
    return { ec, ticket };
}

void impl::bind_host(const std::string& host, const std::string& service)
{
    check_not_started();
    auto& services_by_id = settings_.target_services_by_id;
    // Not performance-critical as it's one time operation.
    if (std::find_if(services_by_id.begin(), services_by_id.end(), [&service](auto& p) {
            return p.second == service;
        }) == services_by_id.end())
    {
        throw std::runtime_error("unknown service " + service);
    }
    if (!settings_.tvm_service_by_host.insert(std::make_pair(host, service)).second)
    {
        throw std::runtime_error("host " + host + " already configured");
    }
}

void impl::check_not_started()
{
    if (started_) throw std::runtime_error("can't subscribe after tvm module start");
}

}}

#include <yplatform/module_registration.h>
REGISTER_MODULE(ymod_tvm::tvm2::impl)
